LOCAL INCLUDE 'GFORT'
      INTEGER   IDUM(10)
      LOGICAL   LDUM(10)
      REAL      RDUM(10)
      DOUBLE PRECISION DDUM(5)
      EQUIVALENCE (DDUM, RDUM, IDUM, LDUM)
      COMMON /TECORG/ DDUM
LOCAL END
LOCAL INCLUDE 'TETABLE.INC'
      INTEGER   TEBUFF(512), TEKOLS(16), TENUMV(16), ITERNO, INDISK,
     *   INCNO
      LOGICAL   DOTABL
      COMMON /TVTABL/ TEBUFf, TEKOLS, TENUMV, ITERNO, DOTABL, INDISK,
     *   INCNO
LOCAL END
      PROGRAM TECOR
C-----------------------------------------------------------------------
C! Use TEC maps to calibrate ionospheric effects
C# Task Calibration VLA VLBI
C-----------------------------------------------------------------------
C;  Copyright (C) 1998-2000, 2001, 2003, 2005, 2010-2011, 2013, 2015,
C;  Copyright (C) 2019, 2022-2025
C;  Associated Universities, Inc. Washington DC, USA.
C;
C;  This program is free software; you can redistribute it and/or
C;  modify it under the terms of the GNU General Public License as
C;  published by the Free Software Foundation; either version 2 of
C;  the License, or (at your option) any later version.
C;
C;  This program is distributed in the hope that it will be useful,
C;  but WITHOUT ANY WARRANTY; without even the implied warranty of
C;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
C;  GNU General Public License for more details.
C;
C;  You should have received a copy of the GNU General Public
C;  License along with this program; if not, write to the Free
C;  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge,
C;  MA 02139, USA.
C;
C;  Correspondence concerning AIPS should be addressed as follows:
C;         Internet email: aipsmail@nrao.edu.
C;         Postal address: AIPS Project Office
C;                         National Radio Astronomy Observatory
C;                         520 Edgemont Road
C;                         Charlottesville, VA 22903-2475 USA
C-----------------------------------------------------------------------
C   TECOR uses maps of ionospheric total electron content (TEC) to
C   determine corrections for ionospheric Faraday rotation and, at the
C   user's option, dispersive delay.
C-----------------------------------------------------------------------
      INCLUDE 'INCS:PUVD.INC'
C
C   Local Variables
C
C      IRET    Subroutine return status
C      UVFILE  Name of UVDATA object used to access AIPS data file
C      ICLTAB  Name of TABLE object used to access input CL table
C      OCLTAB  Name of TABLE object used to access output CL table
C      INPUTS  Name of INPUTS object used to access input adverbs
C      SUBARR  Selected subarray number or zero
C      NANTS   Number of antenna selections
C      ANTENS  List of selected antennas
C      ANTREJ  Should antenna in ANTENS be rejected?
C      DODDEL  Should dispersive delay corrections be calculated?
C      LFACT   Follow rotation of ionosphere (1) to not at all (0)
C      TECFAC  Scale the TEC factor (0 -> 1, Petrov value 0.85)
C      OUTVER  Version number of output CL table
C      NUMIF   Number of IFs in CL tables
C      NUMPOL  Number of polarizations in CL tables
C      NUMMOD  Number of CL rows that were mdofied
C      WORK    Workspace for DIE
C
      INTEGER   IRET
C
      CHARACTER UVFILE*7, ICLTAB*14, OCLTAB*15, INPUTS*13, TECTYP*8
      PARAMETER (UVFILE = 'UV file')
      PARAMETER (ICLTAB = 'Input CL table')
      PARAMETER (OCLTAB = 'Output CL table')
      PARAMETER (INPUTS = 'Input adverbs')
C
      INTEGER   SUBARR, NANTS
      INTEGER   ANTENS(MAXANT)
      LOGICAL   ANTREJ
C
      LOGICAL   DODDEL, DOIGRF
      REAL      LFACT, TECFAC(3)
C
      INTEGER   OUTVER, NUMIF, NUMPOL, TEVER
C
      INTEGER   NUMMOD
      INTEGER   WORK(256)
C
      INCLUDE 'INCS:DDCH.INC'
      INCLUDE 'INCS:DFIL.INC'
      INCLUDE 'INCS:DHDR.INC'
      INCLUDE 'INCS:DHIS.INC'
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'INCS:DUVH.INC'
C-----------------------------------------------------------------------
      CALL TECINI (ICLTAB, OCLTAB, UVFILE, INPUTS, SUBARR, NANTS,
     *   ANTENS, ANTREJ, DODDEL, DOIGRF, TECFAC, LFACT, OUTVER, NUMIF,
     *   NUMPOL, TECTYP, IRET)
C                                       IRET = 0 implies SUBARR >= 0
C                                       IRET = 0 implies
C                                                0 <= NANTS <= MAXANT
C                                       IRET = 0 implies
C                                                ANTENS(1:NANTS) > 0
C                                       IRET = 0 implies OUTVER >= 2
C                                       IRET = 0 implies
C                                                1 <= NUMIF <= MAXIF
C                                       IRET = 0 implies
C                                                1 <= NUMPOL <= 2
      IF (IRET .EQ. 0) THEN
         CALL TECPRO (ICLTAB, OCLTAB, UVFILE, SUBARR, NANTS, ANTENS,
     *      ANTREJ, DODDEL, DOIGRF, TECFAC, LFACT, NUMIF, NUMPOL,
     *      NUMMOD, TEVER, TECTYP, IRET)
C                                       IRET = 0 implies NUMMOD >= 0
         IF (IRET .EQ. 0) THEN
            CALL TECHIS (OCLTAB, INPUTS, DODDEL, DOIGRF, TECFAC, LFACT,
     *         OUTVER, TEVER, NUMMOD, IRET)
            IF (IRET .NE. 0) THEN
               MSGTXT = 'TECOR: FAILED TO UPDATE HISTORY FILE'
               CALL MSGWRT (9)
            END IF
         ELSE
            MSGTXT = 'TECOR: FAILED TO GENERATE CORRECTIONS'
            CALL MSGWRT (9)
         END IF
      ELSE
         MSGTXT = 'TECOR: FAILED TO INITIALIZE PROGRAM'
         CALL MSGWRT (9)
      END IF
C
      CALL DIE (IRET, WORK)
      END
      SUBROUTINE TECINI (ICLTAB, OCLTAB, UVFILE, INPUTS, SUBARR, NANTS,
     *   ANTENS, ANTREJ, DODDEL, DOIGRF, TECFAC, LFACT, OUTVER, NUMIF,
     *   NUMPOL, TECTYP, IRET)
C----------------------------------------------------------------------
C   Read the input adverbs and restart AIPS; open the input and output
C   calibration tables and initialize the module that interprets IONEX
C   files.
C
C   Inputs:
C      ICLTAB   C*(*)       The name of the TABLE object used to access
C                            the input CL table
C      OCLTAB   C*(*)       The name of the TABLE object used to access
C                            the output CL table
C      UVFILE   C*(*)       The name of the UVDATA object used to
C                            access the AIPS data file
C      INPUTS   C*(*)       The name of the INPUTS data object used to
C                            access the input adverbs
C
C   Outputs:
C      SUBARR   I           The number of the subarray to correct or
C                            zero if all subarrays are to be corrected
C      NANTS    I           The number of antennas in the selection
C                            list
C      ANTENS   I(*)        The antenna selection list
C      ANTREJ   L           Are antennas in the selection list to be
C                            rejected?
C      DODDEL   L           Are dispersive delay corrections to be
C                            calculated?
C      DOIGRF   L           Is the IGRF v13 model to be used
C      TECFAC   R           Scale TEC factor
C      LFACT    R           Follow rotation of ionsphere factor
C      OUTVER   I           Version number of the output CL table
C      NUMIF    I           Number of IFs in the CL tables
C      NUMPOL   I           Number of polarizations in the CL tables
C      IRET     I           Return status
C                            0 - task initialized
C                            1 - failed to initialize output table
C                            2 - failed to initialize IONEX module
C                            3 - failed to initialize dispersive
C                                 delay flag
C                            4 - failed to initialize antenna list
C                            5 - failed to initialize input table
C                            6 - failed to read adverbs
C
C   Preconditions:
C      ICLTAB is not blank and is unique
C      OCLTAB is not blank and is unique
C      UVFILE is not blank and is unique
C      INPUTS is not blank and is unique
C      ANTENS has at least MAXANT entries
C
C   Postconditions
C      IRET = 0 implies ICLTAB is open read-only
C      IRET = 0 implies OCLTAB is open for writing and is empty
C      IRET = 0 implies UVFILE has been initialized
C      IRET = 0 implies INPUTS has been initialized
C      IRET = 0 implies SUBARR >= 0
C      IRET = 0 implies 0 <= NANTS <= MAXANT
C      IRET = 0 implies ANTENS(1:NANTS) > 0
C      IRET = 0 implies OUTVER is the version number of OCLTAB
C      IRET = 0 implies OUTVER > 1
C      IRET = 0 implies NUMIF is the number of IFs in ICLTAB and OCLTAB
C      IRET = 0 implies NUMPOL is the number of polarizations in ICLTAB
C                       and OCLTAB
C-----------------------------------------------------------------------
      CHARACTER ICLTAB*(*), OCLTAB*(*), UVFILE*(*), INPUTS*(*), TECTYP*8
      INTEGER   SUBARR, NANTS, ANTENS(*)
      LOGICAL   ANTREJ, DODDEL, DOIGRF
      REAL      LFACT, TECFAC(3)
      INTEGER   OUTVER, NUMIF, NUMPOL, IRET
C
      INCLUDE 'INCS:PAOOF.INC'
C
C     Local variables
C
C     TASKNM    task name
C     NPARMS    number of adverbs
C     AVNAME    list of adverb names
C     AVTYPE    list of adverb type codes
C     AVDIM     list of adverb dimensions
C     FIRSCL    time of earliest record in ICLTAB
C     LASTCL    time of latest record in ICLTAB
C     NUMANT    number of antennas listed in ICLTAB
C     NTERM     number of polynomial delay terms in ICLTAB
C     GMMOD     mean gain modulus of corrections in ICLTAB
C     IRET2     temporary status value
C
      CHARACTER TASKNM*6
      PARAMETER (TASKNM = 'TECOR ')
      INTEGER   NPARMS
      PARAMETER (NPARMS = 12)
      CHARACTER AVNAME(NPARMS)*8
      INTEGER   AVTYPE(NPARMS), AVDIM(2, NPARMS)
C
      DOUBLE PRECISION FIRSCL, LASTCL
      INTEGER   NUMANT, NTERM
      REAL      GMMOD
      INTEGER   IRET2
C
      INCLUDE 'INCS:DMSG.INC'
C
      DATA AVNAME /'INNAME  ', 'INCLASS ', 'INSEQ   ', 'INDISK  ',
     *             'INFILE  ', 'NFILES  ', 'SUBARR  ', 'ANTENNAS',
     *             'GAINVER ', 'GAINUSE ', 'APARM   ', 'DOTABLE '/
      DATA AVTYPE /OOACAR, OOACAR, OOAINT, OOAINT,
     *             OOACAR, OOAINT, OOAINT, OOAINT,
     *             OOAINT, OOAINT, OOARE,  OOARE /
      DATA AVDIM / 12, 1,    6, 1,    1, 1,   1, 1,
     *             48, 1,    1, 1,    1, 1,   50, 1,
     *              1, 1,    1, 1,   10, 1,   1, 1/
C-----------------------------------------------------------------------
      CALL AV2INP (TASKNM, NPARMS, AVNAME, AVTYPE, AVDIM, INPUTS, IRET)
C                                       IRET = 0 implies
C                                                INPUTS initialized
      IF (IRET .EQ. 0) THEN
         CALL TECITB (INPUTS, UVFILE, ICLTAB, NUMANT, NUMIF, NUMPOL,
     *                NTERM, GMMOD, FIRSCL, LASTCL, IRET)
C                                       IRET = 0 implies
C                                                UVFILE initialized
C                                       IRET = 0 implies
C                                                ICLTAB open read-only
C                                       IRET = 0 implies NUMANT is
C                                                number of antennas in
C                                                ICLTAB
C                                       IRET = 0 implies NUMIF is
C                                                number of IFs in
C                                                ICLTAB
C                                       IRET = 0 implies
C                                                1 <= NUMIF <= MAXIF
C                                       IRET = 0 implies NUMPOL is
C                                                number of polarizations
C                                                in ICLTAB
C                                       IRET = 0 implies
C                                                1 <= NUMPOL <= 2
C                                       IRET = 0 implies GMMOD is mean
C                                                gain modulus for
C                                                ICLTAB
C                                       IRET = 0 implies FIRSCL is
C                                                earliest time in ICLTAB
C                                       IRET = 0 implies LASTCL is
C                                                latest time in ICLTAB
         IF (IRET .EQ. 0) THEN
            CALL TECANT (INPUTS, SUBARR, NANTS, ANTENS, ANTREJ, IRET)
C                                       IRET = 0 implies SUBARR >= 0
C                                       IRET = 0 implies
C                                                0 <= NANTS <= MAXANT
C                                       IRET = 0 implies
C                                                ANTENS(1:NANTS) > 0
            IF (IRET .EQ. 0) THEN
               CALL TECDEL (INPUTS, DODDEL, LFACT, DOIGRF, TECFAC, IRET)
               IF (IRET .EQ. 0) THEN
                  CALL TECION (INPUTS, FIRSCL, LASTCL, TECTYP, IRET)
C                                       IRET = 0 implies IONEX module
C                                                initialized
C                                       IRET = 0 implies earliest TEC
C                                                map precedes FIRSCL
C                                       IRET = 0 implies latest TEC
C                                                map follows LASTCL
                  IF (IRET .EQ. 0) THEN
                     CALL TECOTB (INPUTS, UVFILE, OCLTAB, NUMANT, NUMIF,
     *                            NUMPOL, NTERM, GMMOD, OUTVER, IRET)
C                                       IRET = 0 implies OCLTAB is open
C                                                for writing
C                                       IRET = 0 implies OCLTAB is
C                                                empty
C                                       IRET = 0 implies OUTVER >= 2
C                                       IRET = 0 implies OCLTAB has
C                                                NUMANT antennas
C                                       IRET = 0 implies OCLTAB has
C                                                NUMIF IFs
C                                       IRET = 0 implies OCLTAB has
C                                                NUMPOL polarizations
C                                       IRET = 0 implies OCLTAB has
C                                                NTERM polynomial terms
C                                       IRET = 0 implies OCLTAB has a
C                                                mean gain modulus of
C                                                GMMOD
                     IF (IRET .NE. 0) THEN
                        WRITE (MSGTXT, 9001) IRET
                        CALL MSGWRT (9)
                        IRET = 1
                     END IF
                  ELSE
                     WRITE (MSGTXT, 9002) IRET
                     CALL MSGWRT (9)
                     IRET = 2
                  END IF
               ELSE
                  WRITE (MSGTXT, 9003) IRET
                  CALL MSGWRT (9)
                  IRET = 3
               END IF
            ELSE
               WRITE (MSGTXT, 9004) IRET
               CALL MSGWRT (9)
               IRET = 4
            END IF
C
C           Force closure of input table if anything is wrong:
C
            IF (IRET .NE. 0) THEN
               CALL TABCLO (ICLTAB, IRET2)
            END IF
         ELSE
            WRITE (MSGTXT, 9005) IRET
            CALL MSGWRT (9)
            IRET = 5
         END IF
      ELSE
         WRITE (MSGTXT, 9006) IRET
         CALL MSGWRT (9)
         IRET = 6
      END IF
C-----------------------------------------------------------------------
 9001 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING OUTPUT CL TABLE')
 9002 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING IONEX MODULE')
 9003 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING DELAY FLAG')
 9004 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING ANTENNA LIST')
 9005 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING INPUT CL TABLE')
 9006 FORMAT ('TECINI: ERROR ', I4, ' INITIALIZING TASK')
      END
      SUBROUTINE TECITB (INPUTS, UVFILE, ICLTAB, NUMANT, NUMIF, NUMPOL,
     *                   NTERM, GMMOD, FIRSCL, LASTCL, IRET)
C-----------------------------------------------------------------------
C   Initialize the input CL table.
C
C   Inputs:
C      INPUTS   C*(*)       Name of INPUTS object used to access adverbs
C      UVFILE   C*(*)       Name of UVDATA object used to access AIPS
C                            data file
C      ICLTAB   C*(*)       Name of TABLE object used to access input CL
C                            table
C
C   Outputs:
C      NUMANT   I           Number of antennas in CL table
C      NUMIF    I           Number of IFs in CL table
C      NUMPOL   I           Number of polarizations in CL table
C      NTERM    I           Number of polynomial terms in CL table
C      GMMOD    R           Mean gain modulus for CL table
C      FIRSCL   D           Earliest time in CL table (Julian date)
C      LASTCL   D           Latest time in CL table (Julian date)
C      IRET     I           Return status
C                              0 - table initialized
C                              1 - error reading table
C                              2 - error opening table
C                              3 - error obtaining reference date
C                            999 - logic flaw
C-----------------------------------------------------------------------
      CHARACTER INPUTS*(*), UVFILE*(*), ICLTAB*(*)
      INTEGER   NUMANT, NUMIF, NUMPOL, NTERM
      REAL      GMMOD
      DOUBLE PRECISION FIRSCL, LASTCL
      INTEGER   IRET
C
      INCLUDE 'INCS:PAOOF.INC'
      INCLUDE 'INCS:PUVD.INC'
C
C     Local variables
C
C     NUMKEY      Number of UVDATA keywords to initialize from adverbs
C     INKEY       List of adverbs used to initialize UVDATA object
C     OUTKEY      List of UVDATA keywords to receive adverb values
C     INVER       Input table version number
C     NUMROW      Number of rows in CL table
C     CLROW       Current row in CL table
C     TIME        CL row time
C     TIMEI       CL row time interval
C     SOURID      CL row source ID
C     ANTNO       CL row antenna number
C     SUBA        CL row subarray number
C     FREQID      CL row frequency ID
C     IFR         CL row ionospheric Faraday rotation
C     GEODLY      CL row geometric delay polynomial
C     DOPOFF      CL row Doppler offsets
C     ATMOS       CL row atmospheric group delays
C     DATMOS      CL row atmospheric group delay derivatives
C     MBDLY       CL row multiband delays
C     CLOCK       CL row clock offsets
C     DCLOCK      CL row clock drift rates
C     DISP        CL row dispersive delays
C     DDISP       CL row dispersive delay rates
C     CREAL       CL row real gains
C     CIMAG       CL row imaginary gains
C     DELAY       CL row delays
C     RATE        CL row rates
C     WEIGHT      CL row weights
C     REFA        CL row reference antennas
C     DATOBS      Observing date
C     JD0         Julian date at midnight on observing date
C     TYPE        Attribute type code
C     DIM         Attribute dimensions
C     CDUMMY      Dummy character argument
C     IRET2       Temporary subroutine exit status
C
      INTEGER   NUMKEY
      PARAMETER (NUMKEY = 4)
      CHARACTER INKEY(NUMKEY)*8, OUTKEY(NUMKEY)*16
      INTEGER   INVER, NUMROW, CLROW
      DOUBLE PRECISION TIME
      REAL      TIMEI
      INTEGER   SOURID, ANTNO, SUBA, FREQID
      REAL      IFR
      DOUBLE PRECISION GEODLY(12)
      REAL      DOPOFF(MAXIF), ATMOS(2), DATMOS(2), MBDLY(2)
      REAL      CLOCK(2), DCLOCK(2), DISP(2), DDISP(2)
      REAL      CREAL(2, MAXIF), CIMAG(2, MAXIF)
      REAL      DELAY(2, MAXIF), RATE(2, MAXIF), WEIGHT(2, MAXIF)
      INTEGER   REFA(2, MAXIF)
C
      CHARACTER DATOBS*8
      DOUBLE PRECISION JD0
C
      INTEGER   TYPE, DIM(3), IRET2
      CHARACTER CDUMMY
C
      INCLUDE 'GFORT'
      INCLUDE 'TETABLE.INC'
      INCLUDE 'INCS:DMSG.INC'
C
      DATA INKEY /'INNAME  ', 'INCLASS ', 'INSEQ   ', 'INDISK  '/
      DATA OUTKEY /'FILE_NAME.NAME  ', 'FILE_NAME.CLASS ',
     *             'FILE_NAME.IMSEQ ', 'FILE_NAME.DISK'/
C-----------------------------------------------------------------------
      CALL OUVCRE (UVFILE, IRET)
      CALL CHECK ('TECITB', 1, IRET .EQ. 0, IRET)
      IF (IRET .NE. 0) GO TO 999
      CALL IN2OBJ (INPUTS, NUMKEY, INKEY, OUTKEY, UVFILE, IRET)
      CALL CHECK ('TECITB', 2, IRET .EQ. 0, IRET)
      IF (IRET .NE. 0) GO TO 999
      CALL INGET (INPUTS, 'DOTABLE', TYPE, DIM, IDUM, CDUMMY, IRET)
      DOTABL = RDUM(1).GT.0.0
      CALL FILL (15, 0, TENUMV)
      CALL INGET (INPUTS, 'GAINVER', TYPE, DIM, IDUM, CDUMMY, IRET)
      INVER = IDUM(1)
      CALL CHECK ('TECITB', 3, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                         .AND. (DIM(1) .EQ. 1)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
C
C     Override negative values of INVER:
C
      IF (INVER .LT. 0) THEN
         INVER = 0
      END IF
C                                       INVER >= 0
      CALL UV2TAB (UVFILE, ICLTAB, 'CL', INVER, IRET)
      CALL CHECK ('TECITB', 4, IRET .EQ. 0, IRET)
      IF (IRET .NE. 0) GO TO 999
      CALL OCLINI (ICLTAB, 'READ', CLROW, NUMANT, NUMPOL, NUMIF, NTERM,
     *             GMMOD, IRET)
C                                       IRET = 0 implies CLROW = 1
C                                       IRET = 0 implies NUMANT >= 0
C                                       IRET = 0 implies
C                                                1 <= NUMPOL <= 2
C                                       IRET = 0 implies
C                                                1 <= NUMIF <= MAXIF
C                                       IRET = 0 implies NTERM >= 0
      IF (IRET .EQ. 0) THEN
         CALL TABGET (ICLTAB, 'VER', TYPE, DIM, IDUM, CDUMMY, IRET)
         INVER = IDUM(1)
         CALL CHECK ('TECITB', 5, (IRET .EQ. 0)
     *                            .AND. (TYPE .EQ. OOAINT)
     *                            .AND. (DIM(1) .EQ. 1)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         CALL INPUTT (INPUTS, 'GAINVER', TYPE, DIM, IDUM, CDUMMY, IRET)
         WRITE (MSGTXT, 1000) INVER
         CALL MSGWRT (5)
         WRITE (MSGTXT, 1001)
         CALL MSGWRT (5)
         CALL TABGET (ICLTAB, 'NROW', TYPE, DIM, IDUM, CDUMMY, IRET)
         NUMROW = IDUM(1)
         CALL CHECK ('TECITB', 6, (IRET .EQ. 0)
     *                            .AND. (TYPE .EQ. OOAINT)
     *                            .AND. (DIM(1) .EQ. 1)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         FIRSCL = +9999.0D0
         LASTCL = -9999.0D0
C
C        Invariant: IRET = 0 implies FIRSCL is the earliest unflagged
C                            time in the first CLROW - 1 rows of ICLTAB
C                            and LASTCL is the lastest unflagged time
C                            in the first CLROW -1 rows of ICLTAB
C        Bound: NUMROW - CLROW + 1
C
   10    IF ((IRET .EQ. 0) .AND. (CLROW .NE. (NUMROW + 1))) THEN
            CALL OTABCL (ICLTAB, 'READ', CLROW, NUMPOL, NUMIF, TIME,
     *                   TIMEI, SOURID, ANTNO, SUBA, FREQID, IFR,
     *                   GEODLY, DOPOFF, ATMOS, DATMOS, MBDLY, CLOCK,
     *                   DCLOCK, DISP, DDISP, CREAL, CIMAG, DELAY, RATE,
     *                   WEIGHT, REFA, IRET)
            IF (IRET .EQ. 0) THEN
               IF (TIME .LT. FIRSCL) THEN
                  FIRSCL = TIME
               END IF
               IF (TIME .GT. LASTCL) THEN
                  LASTCL = TIME
               END IF
            ELSE IF (IRET .EQ. -1) THEN
C
C              Ignore flagged records:
C
               IRET = 0
            ELSE
               WRITE (MSGTXT, 9010) IRET
               CALL MSGWRT (9)
               IRET = 1
               CALL OTABCL (ICLTAB, 'CLOS', CLROW, NUMPOL, NUMIF, TIME,
     *                      TIMEI, SOURID, ANTNO, SUBA, FREQID, IFR,
     *                      GEODLY, DOPOFF, ATMOS, DATMOS, MBDLY, CLOCK,
     *                      DCLOCK, DISP, DDISP, CREAL, CIMAG, DELAY,
     *                      RATE, WEIGHT, REFA, IRET2)
            END IF
            GO TO 10
         END IF
         WRITE (MSGTXT, 1010)
         CALL MSGWRT (5)
      ELSE
         WRITE (MSGTXT, 9011) IRET
         CALL MSGWRT (9)
         IRET = 2
      END IF
C
C     Add Julian day offset to times
C
      CALL OUVATT (UVFILE, .FALSE., IRET)
      IF (IRET. EQ. 0) THEN
         CALL UVDGET (UVFILE, 'DATE-OBS', TYPE, DIM, IDUM, DATOBS, IRET)
         CALL CHECK ('TECITB', 7, (IRET .EQ. 0) .AND. (TYPE .EQ. OOACAR)
     *                            .AND. (DIM(1) .EQ. 8)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL JULDAY (DATOBS, JD0)
         FIRSCL = FIRSCL + JD0
         LASTCL = LASTCL + JD0
      ELSE
         WRITE (MSGTXT, 9012) IRET
         CALL MSGWRT (9)
         IRET = 3
      END IF
C
  999 RETURN
C-----------------------------------------------------------------------
 1000 FORMAT ('Reading from CL table version number ', I4)
 1001 FORMAT ('Searching for earliest and latest times in CL table')
 1010 FORMAT ('Done')
 9010 FORMAT ('TECITB: ERROR ', I4, ' READING CL TABLE')
 9011 FORMAT ('TECITB: ERROR ', I4, ' OPENING CL TABLE')
 9012 FORMAT ('TECITB: ERROR ', I4, ' ATTACHING UV FILE')
      END
      SUBROUTINE TECANT (INPUTS, SUBARR, NANTS, ANTENS, ANTREJ, IRET)
C-----------------------------------------------------------------------
C   Initialize antenna selection.
C
C   Inputs:
C      INPUTS  C*(*)      Name of INPUTS object used to access adverbs
C
C   Outputs:
C      NANTS   I          Number of items in antenna selection list
C      ANTENS  I(*)       Antenna selection list
C      ANTREJ  L          Should antennas in selection list be rejected
C      IRET    I          Return status
C                            0 - selection initalized
C                          999 - logic flaw
C
C   Preconditions:
C      INPUTS is initialized
C      ANTENS has at least MAXANT elements
C
C   Postconditions:
C      IRET = 0 implies SUBARR >= 0
C      IRET = 0 implies 0 <= NANTS <= MAXANT
C      IRET = 0 implies ANTENS(1:NANTS) > 0
C-----------------------------------------------------------------------
      CHARACTER INPUTS*(*)
      INTEGER   SUBARR, NANTS, ANTENS(*)
      LOGICAL   ANTREJ
      INTEGER   IRET
C
      INCLUDE 'INCS:PAOOF.INC'
      INCLUDE 'INCS:PUVD.INC'
C
C     Local Variables:
C
C     ANTLIS    Values of ANTENNAS adverb
C     TYPE      Attribute type
C     DIM       Attribute dimensions
C     I         Array index
C     CDUMMY    Dummy character argument
C
      INTEGER   ANTLIS(50), TYPE, DIM(3), I
      CHARACTER CDUMMY
C
      INCLUDE 'GFORT'
      INCLUDE 'INCS:DMSG.INC'
C-----------------------------------------------------------------------
      CALL INGET (INPUTS, 'SUBARR', TYPE, DIM, IDUM, CDUMMY, IRET)
      SUBARR = IDUM(1)
      CALL CHECK ('TECANT', 1, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                         .AND. (DIM(1) .EQ. 1)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
      IF (SUBARR .LT. 0) THEN
         WRITE (MSGTXT, 1000)
         CALL MSGWRT (5)
         SUBARR = 1
      END IF
      CALL INGET (INPUTS, 'ANTENNAS', TYPE, DIM, ANTLIS, CDUMMY, IRET)
      CALL CHECK ('TECANT', 2, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                         .AND. (DIM(1) .LE. 50)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
      NANTS = 0
      ANTREJ = .FALSE.
C
C     Invariant: NANTS is the minimum of MAXANT and the number of
C                non-zero entries in ANTLIS(1:I-1) and ANTENS(1:NANTS)
C                constains the absolute values of the first NANTS
C                non-zero entries in ANTLIS and ANTREJ is true if and
C                only if there is a negative element in ANTLIS(1:I-1).
C
      DO 10 I = 1, DIM(1)
         IF (ANTLIS(I) .NE. 0) THEN
            IF (ANTLIS(I) .LT. 0) THEN
               ANTREJ = .TRUE.
            END IF
            IF (NANTS .LT. MAXANT) THEN
               NANTS = NANTS + 1
               ANTENS(NANTS) = ABS(ANTLIS(I))
            ELSE
               WRITE (MSGTXT, 1001) ABS(ANTLIS(I))
               CALL MSGWRT (5)
            END IF
         END IF
   10 CONTINUE
C
  999 RETURN
C-----------------------------------------------------------------------
 1000 FORMAT ('Defaulting to subarray 1')
 1001 FORMAT ('Ignoring selection of antenna ', I4, ' (no more room)')
      END
      SUBROUTINE TECDEL (INPUTS, DODDEL, LFACT, DOIGRF, TECFAC, IRET)
C-----------------------------------------------------------------------
C   Set the dispersive delay corrections flag.
C
C   Inputs:
C      INPUTS   C*(*)    Name of INPUTS object used to access adverbs
C
C   Outputs:
C      DODDEL   L        Calculate dispersive delay corrections?
C      LFACT    R        Follow ionosphere factor
C      DOIGRF   L        Use IGRF v13 model
C      TECFAC   R        Scale TEC factor
C      IRET     I        Return status
C                           0 - flag set
C                         999 - logic flaw
C-----------------------------------------------------------------------
      CHARACTER INPUTS*(*)
      LOGICAL   DODDEL, DOIGRF
      REAL      LFACT, TECFAC(3)
      INTEGER   IRET
C
      INCLUDE 'INCS:PAOOF.INC'
C
C     Local Variables
C
C     APARM    Value of APARM adverb
C     TYPE     Attribute type
C     DIM      Attribute dimensions
C     CDUMMY   Dummy character argument
C
      REAL      APARM(10)
      INTEGER   TYPE, DIM(3)
      CHARACTER CDUMMY
C
      INCLUDE 'GFORT'
      INCLUDE 'INCS:DMSG.INC'
C-----------------------------------------------------------------------
      CALL INGET (INPUTS, 'APARM', TYPE, DIM, IDUM, CDUMMY, IRET)
      CALL RCOPY (10, RDUM, APARM)
      CALL CHECK ('TECDEL', 1, (IRET .EQ. 0) .AND. (TYPE .EQ. OOARE)
     *                         .AND. (DIM(1) .LE. 10)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
      DODDEL = APARM(1).GT.0.0
      LFACT = APARM(2)
      IF (APARM(2).EQ.0.0) LFACT = 1.0
      LFACT = MAX (0.0, MIN (1.0, LFACT))
      DOIGRF = APARM(3).LE.0.0
      IF ((APARM(4).LT.0.1) .OR. (APARM(4).GT.5)) APARM(4) = 1.0
      TECFAC(1) = APARM(4)
      IF ((APARM(5).LT.-200.) .OR. (APARM(5).GT.200.)) APARM(4) = 0.0
      TECFAC(2) = APARM(5) * 1000.0
      IF ((APARM(6).LT.0.5) .OR. (APARM(6).GT.2.)) APARM(6) = 1.0
      TECFAC(3) = APARM(6)
C
      IF (DODDEL) THEN
         MSGTXT = 'Dispersive delay corrections will be calculated'
      ELSE
         MSGTXT = 'Dispersive delay corrections will not be calculated'
      END IF
      CALL MSGWRT (5)
C
  999 RETURN
      END
      SUBROUTINE TECION (INPUTS, FIRSCL, LASTCL, TECTYP, IRET)
C-----------------------------------------------------------------------
C   Initialize the IONEX module and check that the TEC data spans the
C   times covered by the CL table.
C
C   Inputs:
C      INPUTS  C*(*)     Name of the INPUTS object used to access the
C                         adverb values
C      FIRSCL  D         Earliest time in the CL table
C      LASTCL  D         Latest time in the CL table
C
C   Outputs:
C      TECTYP  C*8       TEC data type
C      IRET    I         Return status
C                           0 - IONEX module initialized
C                           1 - TEC data does not cover time range
C                           2 - failed to initialize IONEX
C                           3-  IONEX filename incorrect format (if NFILES
C                               > 1)
C                         999 - logic flaw
C
C   Preconditions:
C      INPUTS is initialized
C
C   Postconditions:
C      IRET = 0 imples IONEX module is initialized
C      IRET = 0 implies IONFIR() <= FIRSCL
C      IRET = 0 implies IONLAS() >= LASTCL
C-----------------------------------------------------------------------
      CHARACTER INPUTS*(*), TECTYP*8
      DOUBLE PRECISION FIRSCL, LASTCL
      INTEGER   IRET
C
      INCLUDE 'INCS:PAOOF.INC'
C
C     Local variables
C
C     INFILE     Inputs file name
C     TYPE       Attribute type code
C     DIM        Attribute dimensions
C     NFILES     Number if IONEX files
C     DAYNUM     Day number to be used in the names of IONEX files
C     YEAR       Year to be used in the names of IONEX files
C     CHDAYN     Character day number
C     CHYEAR     Character 2 digit year
C
      CHARACTER INFILE*48, FILENA*48, CHYEAR*2, CHDAYN*3,
     *CDUMMY
      INTEGER   TYPE, DIM(3), DAYNUM, YEAR, NFILES, I, J, K, JTRIM,
     *   LEAP
C
      DOUBLE PRECISION IONFIR, IONLAS
      EXTERNAL  IONFIR, IONLAS
C
      INCLUDE 'GFORT'
      INCLUDE 'INCS:DMSG.INC'
C-----------------------------------------------------------------------
      CALL INGET (INPUTS, 'INFILE', TYPE, DIM, IDUM, INFILE, IRET)
      CALL CHECK ('TECION', 1, (IRET .EQ. 0) .AND. (TYPE .EQ. OOACAR)
     *                         .AND. (DIM(1) .LE. 48)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      CALL INGET (INPUTS, 'NFILES', TYPE, DIM, IDUM, CDUMMY, IRET)
      NFILES = IDUM(1)
      CALL CHECK ('TECION', 2, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                         .AND. (DIM(1) .LE. 1)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      LEAP = 0
      IF (IRET .NE. 0) GO TO 999
      J = JTRIM (INFILE)
      IF (INFILE(J-1:J).EQ.'.Z') INFILE(J-1:) = ' '
      IF (INFILE(J-2:J).EQ.'.gz') INFILE(J-2:) = ' '
      J = INDEX(INFILE, ':')
      IF (J .EQ. 0) THEN
         DO 5 K=48, 1, -1
            IF (INFILE(K:K) .EQ. '/') THEN
               J = K
               GO TO 7
               END IF
 5          CONTINUE
         END IF
 7    TECTYP = INFILE(J+1:J+4)
      IF (NFILES .GT. 1) THEN
C                                             Find daynum and year
         READ(INFILE(J+5:J+7),'(I3)',ERR=20) DAYNUM
         READ(INFILE(J+10:J+11),'(I2)',ERR=20) YEAR
C                                             Check for leap year, next divisiable
C                                             by 100 non-leap-year is 2100
C                                             last one was 1900, neither a worry
         IF (MOD(YEAR,4).EQ.0) LEAP=1
C                                             Check valid daynum and year
         IF (DAYNUM .GT. 365+LEAP .OR. DAYNUM .LT. 0 .OR.
     *         YEAR .LT. 0) THEN
            WRITE (MSGTXT, 9002)
            CALL MSGWRT (9)
            IRET = 3
            GO TO 999
         END IF
C                                             Loop through files
         DO 10 I=1, NFILES
C                                             Increment year if DAYNUM > 365
           IF (DAYNUM .GT. 365+LEAP) THEN
              DAYNUM = DAYNUM-365-LEAP
              YEAR = YEAR+1
           ENDIF
C                                             Convert daynum and year to CHAR
           WRITE(CHDAYN,'(I3)') DAYNUM
           WRITE(CHYEAR,'(I2)') YEAR
           IF (YEAR .LT. 10) CHYEAR(1:1) = '0'
           IF (DAYNUM .LT. 100) CHDAYN(1:1) = '0'
           IF (DAYNUM .LT. 10)  CHDAYN(2:2) = '0'
C                                             Create filename
           FILENA = INFILE(1:J+4)//CHDAYN//INFILE(J+8:J+9)
     *//CHYEAR//INFILE(J+12:J+12)
           CALL IONINI (FILENA, I, IRET)
           DAYNUM = DAYNUM+1
   10    CONTINUE
      ELSE
         CALL IONINI (INFILE, 1, IRET)
      END IF
      IF (IRET .EQ. 0) THEN
         IF ((IONFIR() .GT. FIRSCL) .OR. (IONLAS() .LE. LASTCL)) THEN
            WRITE (MSGTXT, 9000)
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9001) IRET
         CALL MSGWRT (9)
         IRET = 2
      END IF
      GO TO 999
C                               File format incorrect
   20 WRITE (MSGTXT, 9002)
      CALL MSGWRT (9)
      IRET=3
C
  999 RETURN
C-----------------------------------------------------------------------
 9000 FORMAT ('TECION: TEC DATA DOES NOT COVER FULL RANGE OF CL TABLE')
 9001 FORMAT ('TECION: ERROR ', I4, ' INITIALIZING IONEX MODULE')
 9002 FORMAT ('TECION: INVALID DAY NUMBER AND/OR YEAR')
      END
      SUBROUTINE TECOTB (INPUTS, UVFILE, OCLTAB, NUMANT, NUMIF, NUMPOL,
     *                   NTERM, GMMOD, OUTVER, IRET)
C-----------------------------------------------------------------------
C   Open the output CL table.
C
C   Inputs:
C      INPUTS   C*(*)   Name of the INPUTS object used to access adverbs
C      UVFILE   C*(*)   Name of the UVDATA object used to access the
C                        AIPS data file
C      OCLTAB   C*(*)   Name of the TABLE object used to access the
C                        output CL table
C      NUMANT   I       Number of antennas for CL table
C      NUMIF    I       Number of IFs for CL table
C      NUMPOL   I       Number of polarizations for CL table
C      NTERM    I       Number of polynomial terms for CL table
C      GMMOD    R       Mean gain modulus for CL table
C
C   Ouputs:
C      OUTVER   I       Version number of output CL table
C      IRET     I       Return status
C                          0 - table opened
C                          1 - output table already exists
C                          2 - can not open output table
C                        999 - logic flaw
C
C   Preconditions:
C      INPUTS is initialized
C      UVFILE is initialized
C      NUMANT >= 0
C      1 <= NUMIF <= MAXIF
C      1 <= NUMPOL <= 2
C      NTERM >= 0
C
C   Postconditions:
C     IRET = 0 implies OCLTAB is open for writing and is empty
C-----------------------------------------------------------------------
      CHARACTER INPUTS*(*), UVFILE*(*), OCLTAB*(*)
      INTEGER   NUMANT, NUMIF, NUMPOL, NTERM
      REAL      GMMOD
      INTEGER   OUTVER, IRET
C
      INCLUDE 'INCS:PAOOF.INC'
C
C     Local variables
C
C     CLROW     CL table row number
C     TYPE      Attribute type code
C     DIM       Attribute dimensions
C     CDUMMY    Dummy character argument
C
      INTEGER   CLROW, TYPE, DIM(3)
      CHARACTER CDUMMY
C
      INCLUDE 'GFORT'
      INCLUDE 'INCS:DMSG.INC'
C-----------------------------------------------------------------------
      CALL INGET (INPUTS, 'GAINUSE', TYPE, DIM, IDUM, CDUMMY, IRET)
      OUTVER = IDUM(1)
      CALL CHECK ('TECOTB', 1, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                         .AND. (DIM(1) .EQ. 1)
     *                         .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
C
C     Fill in default for negative table version numbers:
C
      IF (OUTVER.LT.0) OUTVER = 0
C
      CALL UV2TAB (UVFILE, OCLTAB, 'CL', OUTVER, IRET)
      CALL CHECK ('TECOTB', 2, IRET .EQ. 0, IRET)
      IF (IRET .NE. 0) GO TO 999
C
      CALL OCLINI (OCLTAB, 'WRIT', CLROW, NUMANT, NUMPOL, NUMIF, NTERM,
     *             GMMOD, IRET)
      IF (IRET .EQ. 0) THEN
         IF (CLROW .GT. 1) THEN
C                                       OUTVER > 0 since a new table
C                                       will always be created otherwise
            WRITE (MSGTXT, 9000) OUTVER
            CALL MSGWRT (9)
            IRET = 1
         ELSE
C
C           Find the actual version number:
C
            CALL TABGET (OCLTAB, 'VER', TYPE, DIM, IDUM, CDUMMY, IRET)
            OUTVER = IDUM(1)
            CALL CHECK ('TECOTB', 3, (IRET .EQ. 0)
     *                               .AND. (TYPE .EQ. OOAINT)
     *                               .AND. (DIM(1) .EQ. 1)
     *                               .AND. (DIM(2) .EQ. 1), IRET)
            IF (IRET .NE. 0) GO TO 999
C
            WRITE (MSGTXT, 1000) OUTVER
            CALL MSGWRT (5)
            IDUM(1) = OUTVER
            CALL INPUTT (INPUTS, 'GAINUSE', TYPE, DIM, IDUM, CDUMMY,
     *         IRET)
         END IF
      ELSE
         WRITE (MSGTXT, 9001) IRET
         CALL MSGWRT (9)
         IRET = 2
      END IF
C
  999 RETURN
C-----------------------------------------------------------------------
 1000 FORMAT ('Creating CL table version ', I4)
 9000 FORMAT ('TECOTB: WILL NOT OVERWRITE EXISTING CL TABLE ', I4)
 9001 FORMAT ('TECOTB: ERROR ', I4,
     *        ' OPENING OUTPUT CL TABLE FOR WRITING')
      END
      SUBROUTINE TECPRO (ICLTAB, OCLTAB, UVFILE, SUBARR, NANTS, ANTENS,
     *   ANTREJ, DODDEL, DOIGRF, TECFAC, LFACT, NUMIF, NUMPOL, NUMMOD,
     *   TEVER, TECTYP, IRET)
C-----------------------------------------------------------------------
C   Copy records from ICLTAB to OCLTAB, updating ionospheric Faraday
C   rotation and dispersive delay for those records that meet the
C   subarray and antenna selection criteria.
C
C   Inputs:
C      ICLTAB  C*(*)  Name of the TABLE object used to access the
C                      input CL table
C      OCLTAB  C*(*)  Name of the TABLE object used to access the
C                      output CL table
C      UVFILE  C*(*)  Name of the UVDATA object used to access the AIPS
C                      data file
C      SUBARR  I      Selected subarray or 0 if all subarrays are
C                      selected
C      NANTS   I      Number of antennas in selection list
C      ANTENS  I(*)   Antenna selection list
C      ANTREJ  L      Are antennas in selection list to be rejected?
C      DODDEL  L      Should dispersive delays be calculated?
C      LFACT   R      Ionosphere following factor
C      DOIGRF  L      Is the IGRF v13 model to be used
C      TECFAC  R      Scale TEC factor
C      NUMIF   I      Number of IFs in CL tables
C      NUMPOL  I      Number of polarizations in CL table
C
C   Outputs:
C      TEVER   I      TE file version
C      NUMMOD  I      Number of rows modified
C      IRET    I      Return status
C                        0 - table copied
C                        1 - I/O error
C
C   Preconditions:
C      ICLTAB is open read-only
C      OCLTAB is open for writing and is empty
C      UVFILE is initialized
C      SUBARR >= 0
C      0 <= NANTS <= length(ANTENS)
C      0 <= NUMIF <= MAXIF
C      1 <= NUMPOL <= 2
C
C   Postconditions
C      IRET = 0 implies ICLTAB is closed
C      IRET = 0 implies OCLTAB is closed
C      IRET = 0 implies NUMOD >= 0
C-----------------------------------------------------------------------
      CHARACTER ICLTAB*(*), OCLTAB*(*), UVFILE*(*), TECTYP*8
      INTEGER   SUBARR, NANTS, ANTENS(*), TEVER
      LOGICAL   ANTREJ, DODDEL, DOIGRF
      REAL      LFACT, TECFAC(3)
      INTEGER   NUMIF, NUMPOL, NUMMOD, IRET
C
      INCLUDE 'INCS:PAOOF.INC'
      INCLUDE 'INCS:PUVD.INC'
C
C     Local variables
C
C     NUMROW      Number of rows in input table
C     IROW        Input row number
C     OROW        Output row number
C     TIME        CL row time
C     TIMEI       CL row time interval
C     SOURID      CL row source ID
C     ANTNO       CL row antenna number
C     SUBA        CL row subarray number
C     FREQID      CL row frequency ID
C     IFR         CL row ionospheric Faraday rotation
C     GEODLY      CL row geometric delay polynomial
C     DOPOFF      CL row Doppler offsets
C     ATMOS       CL row atmospheric group delays
C     DATMOS      CL row atmospheric group delay derivatives
C     MBDLY       CL row multiband delays
C     CLOCK       CL row clock offsets
C     DCLOCK      CL row clock drift rates
C     DISP        CL row dispersive delays
C     DDISP       CL row dispersive delay rates
C     CREAL       CL row real gains
C     CIMAG       CL row imaginary gains
C     DELAY       CL row delays
C     RATE        CL row rates
C     WEIGHT      CL row weights
C     REFA        CL row reference antennas
C     ANT         Antenna index in ANTENS
C     WANTEN      Is antenna wanted?
C     TYPE        Attribute type code
C     DIM         Attribute dimensions
C     CDUMMY      Dummy character argument
C     IRET2       Return status for table closes
C
      INTEGER   NUMROW, IROW, OROW
C
      DOUBLE PRECISION TIME
      REAL      TIMEI
      INTEGER   SOURID, ANTNO, SUBA, FREQID
      REAL      IFR, DUM, DUM3(3), DUM2(2)
      DOUBLE PRECISION GEODLY(12)
      REAL      DOPOFF(MAXIF), ATMOS(2), DATMOS(2), MBDELY(2)
      REAL      CLOCK(2), DCLOCK(2), DISP(2), DDISP(2)
      REAL      CREAL(2, MAXIF), CIMAG(2, MAXIF)
      REAL      DELAY(2, MAXIF), RATE(2, MAXIF), WEIGHT(2, MAXIF)
      INTEGER   REFA(2, MAXIF)
C
      INTEGER   ANT, I
      LOGICAL   WANTEN
      INTEGER   TYPE, DIM(3)
      CHARACTER CDUMMY
      INTEGER   IRET2
C
      INCLUDE 'GFORT'
      INCLUDE 'TETABLE.INC'
      INCLUDE 'INCS:DMSG.INC'
C-----------------------------------------------------------------------
      NUMMOD = 0
      CALL TABGET (ICLTAB, 'NROW', TYPE, DIM, IDUM, CDUMMY, IRET)
      NUMROW = IDUM(1)
      CALL CHECK ('TECPRO', 1, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *            .AND. (DIM(1) .EQ. 1) .AND. (DIM(2) .EQ. 1), IRET)
      IF (IRET .NE. 0) GO TO 999
C
      IROW = 1
      OROW = 1
C
C     Invariant: IRET = 0 implies IROW - 1 rows have been read from
C                                 ICLTAB
C     Bound: NUMROW + 1 - IROW
C
   10 IF ((IRET .EQ. 0) .AND. (IROW .NE. NUMROW + 1)) THEN
         CALL OTABCL (ICLTAB, 'READ', IROW, NUMPOL, NUMIF, TIME, TIMEI,
     *                SOURID, ANTNO, SUBA, FREQID, IFR, GEODLY, DOPOFF,
     *                ATMOS, DATMOS, MBDELY, CLOCK, DCLOCK, DISP,
     *                DDISP, CREAL, CIMAG, DELAY, RATE, WEIGHT, REFA,
     *                IRET)
         IF (IRET .EQ. 0) THEN
            IF ((SUBARR .EQ. 0) .OR. (SUBA .EQ. SUBARR)) THEN
               IF (NANTS .EQ. 0) THEN
                  WANTEN = .TRUE.
               ELSE
                  WANTEN = .FALSE.
                  DO 20 ANT = 1, NANTS
                     IF (ANTENS(ANT) .EQ. ANTNO) THEN
                        WANTEN = .TRUE.
                     END IF
   20             CONTINUE
                  IF (ANTREJ) THEN
                     WANTEN = .NOT. WANTEN
                  END IF
               END IF
               IF (WANTEN) THEN
                  CALL TECCOR (UVFILE, DODDEL, LFACT, TEVER, TECTYP,
     *               TIME, SUBA, ANTNO, SOURID, NUMPOL, DOIGRF, TECFAC,
     *               IFR, DISP, DDISP, IRET)
C
C                 Ignore failed interpolation:
C
                  IF (IRET .EQ. 1) THEN
                     IRET = 0
                  END IF
                  IF (IRET .EQ. 0) THEN
                     NUMMOD = NUMMOD + 1
                  ELSE
                     WRITE (MSGTXT, 9020) IRET
                     CALL MSGWRT (9)
                     IRET = 1
                  END IF
               END IF
            END IF
            IF (IRET .EQ. 0) THEN
               CALL OTABCL (OCLTAB, 'WRIT', OROW, NUMPOL, NUMIF, TIME,
     *                      TIMEI, SOURID, ANTNO, SUBA, FREQID, IFR,
     *                      GEODLY, DOPOFF, ATMOS, DATMOS, MBDELY,
     *                      CLOCK, DCLOCK, DISP, DDISP, CREAL, CIMAG,
     *                      DELAY, RATE, WEIGHT, REFA, IRET)
               IF (IRET .NE. 0) THEN
                  WRITE (MSGTXT, 9021) IRET
                  CALL MSGWRT (9)
                  IRET = 1
               END IF
            END IF
         ELSE IF (IRET .EQ. -1) THEN
C
C           Skip flagged records:
C
            IRET = 0
         ELSE
            WRITE (MSGTXT, 9022) IRET
            CALL MSGWRT (9)
            IRET = 1
         END IF
         GO TO 10
      END IF
C
C     Close tables:
C
      CALL OTABCL (OCLTAB, 'CLOS', OROW, NUMPOL, NUMIF, TIME, TIMEI,
     *             SOURID, ANTNO, SUBA, FREQID, IFR, GEODLY, DOPOFF,
     *             ATMOS, DATMOS, MBDELY, CLOCK, DCLOCK, DISP, DDISP,
     *             CREAL, CIMAG, DELAY, RATE, WEIGHT, REFA, IRET2)
      IF (IRET2 .EQ. 0) THEN
         CALL OTABCL (ICLTAB, 'CLOS', IROW, NUMPOL, NUMIF, TIME,
     *                TIMEI, SOURID, ANTNO, SUBA, FREQID, IFR,
     *                GEODLY, DOPOFF, ATMOS, DATMOS, MBDELY, CLOCK,
     *                DCLOCK, DISP, DDISP, CREAL, CIMAG, DELAY, RATE,
     *                WEIGHT, REFA, IRET2)
         IF (IRET2 .NE. 0) THEN
            WRITE (MSGTXT, 9023) IRET2
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9024) IRET2
         CALL MSGWRT (9)
         IRET = 1
      END IF
C
      IF (IRET .EQ. 0) THEN
         WRITE (MSGTXT, 1020) OROW - 1
         CALL MSGWRT (5)
         WRITE (MSGTXT, 1021) NUMMOD
         CALL MSGWRT (5)
      END IF
C                                       close TE table
      IF (DOTABL) THEN
         CALL TETABL ('CLOS', I, TECTYP, TIME, SOURID, ANTNO, DUM, DUM,
     *      DUM, DUM, DUM, DUM, DUM, DUM3, DUM, DUM, DUM, DUM, DUM2)
         END IF
C
  999 RETURN
C-----------------------------------------------------------------------
 1020 FORMAT ('Copied ', I6, ' records to the output table')
 1021 FORMAT (I6, ' were modified')
 9020 FORMAT ('TECPRO: ERROR ', I4, ' OBTAINING NEW CALIBRATION')
 9021 FORMAT ('TECPRO: ERROR ', I4, ' WRITING TO OUTPUT TABLE')
 9022 FORMAT ('TECPRO: ERROR ', I4, ' READING FROM INPUT TABLE')
 9023 FORMAT ('TECPRO: ERROR ', I4, ' CLOSING OUTPUT TABLE')
 9024 FORMAT ('TECPRO: ERROR ', I4, ' CLOSING INPUT TABLE')
      END
      SUBROUTINE TECCOR (UVFILE, DODDEL, LFACT, TEVER, TECTYP, TIME,
     *   SUBA, ANTNO, SOURID, NUMPOL, DOIGRF, TECFAC, IFR, DISP, DDISP,
     *   IRET)
C-----------------------------------------------------------------------
C   Calculate ionospheric Faraday rotation and dispersive delay
C   corrections.
C   Inputs:
C      UVFILE  C*(*)     Name of UVDATA object used to access data file
C      DODDEL  L         Calculate dispersive delays?
C      LFACT   R         Follow ionospher factor
C      TIME    D         Time at which to calculate correction
C      SUBA    I         Subarray number
C      ANTNO   I         Antenna number
C      SOURID  I         Source ID number
C      NUMPOL  I         Number of polarizations
C      DOIGRF  L         Is the IGRF v13 model to be used
C      TECFAC  R         Scale TEC factor
C   Outputs:
C      TEVER   I         TE table version
C      IFR     R         Ionospheric Faraday rotation
C      DISP    R(*)      Dispersive delay in seconds at 1m wavelength
C      DDISP   R(*)      Derivative of dispersive delay wrt time in
C                         seconds at 1m wavelength
C      IRET    I         Return status
C                           0 - corrections calculated
C                           1 - TEC data not available
C                           2 - I/O error
C                         999 - logic flaw
C
C   Preconditions:
C      UVFILE is initialized
C      SUBA >= 1
C      1 <= ANTNO <= MAXANT
C      SOURID >= 1
C      1 <= NUMPOL <= 2
C
C   Postconditions:
C      IRET = 1 implies IFR = FBLANK
C      IRET = 1 and DODDEL implies DISP(1:NUMPOL) = FBLANK
C      IRET = 1 and DODDEL implies DDISP(1:NUMPOL) = FBLANK
C-----------------------------------------------------------------------
      CHARACTER UVFILE*(*), TECTYP*8
      LOGICAL   DODDEL, DOIGRF
      DOUBLE PRECISION TIME
      INTEGER   TEVER, SUBA, ANTNO, SOURID, NUMPOL
      REAL      LFACT, TECFAC(3), IFR, DISP(*), DDISP(*)
      INTEGER   IRET
C
      INCLUDE 'INCS:PAOOF.INC'
      INCLUDE 'INCS:PSTD.INC'
      INCLUDE 'INCS:PUVD.INC'
C
C     Local variables:
C
C     LASTSA    Subarray number on last call
C     LASTSU    Source number on last call
C     DISK      Disk number for UV file
C     CNO       Catalogue number for UV file
C     CATBLK    Header block for UV file
C     SULUN     AIPS LUN for SU table
C     ANBUFF    Buffer for reading AN table
C     ANT       Antenna number
C     ANTIDX    Antenna index
C     HA        Hour angle (radians)
C     EL        Elevation (radians)
C     ZA        Zenith angle (radians)
C     COSAZ     Cosine of azimuth
C     AZ        Azimuth (radians)
C     DLAT      Latitude offset of puncture point (radians)
C     DLON      Longitude offset of puncture point (radians)
C     AZION     Azimuth at puncture point (radians)
C     ZAION     Zenith angle at puncture point (radians)
C     TEC       Zenith TEC at puncture point (electrons per square
C                meter)
C     TEPATH    TEC along line of sight
C     B         Magnetic field vector at puncture point
C     MAG       Magnetic field along line of sight
C     RADIUS    Radius of the Earth (meters)
C     POL       Polarization number
C     DATOBS    Reference date for observations
C     JD0       Julian day number on reference date
C     JD0UTC    Julian date corrected to UTC
C     TYPE      Attribute type
C     DIM       Attribute dimensions
C     CDUMMY    Dummy character argument
C
      INTEGER   LASTSA, LASTSU, DISK, CNO, CATBLK(256), SULUN, LASTID
      SAVE      LASTSA, LASTSU, DISK, CNO, CATBLK, SULUN, LASTID
      INTEGER   ANBUFF(512), ANT, ANTIDX
      REAL      HA, EL, ZA, TR, TLAST
      REAL      AZ, DLAT, DLON, AZION, ZAION, TEC, TEPATH, B(3),
     *          MAG, RADIUS, PETROV, DBG
      INTEGER   POL, IDATE(3), DAYN
      CHARACTER DATOBS*8
      DOUBLE PRECISION JD0, JD0UTC, DRA, DDEC, DATE, COLAT, ELONG, X, Y,
     *   Z, F, ALT
      SAVE      JD0, JD0UTC, DRA, DDEC, TLAST, IDATE
      INTEGER   TYPE, DIM(3)
      CHARACTER CDUMMY
      LOGICAL   PLANET
C
      PARAMETER (RADIUS = 6378000.0)
C
      REAL      IONALT
      EXTERNAL  IONALT
C
      INCLUDE 'GFORT'
      INCLUDE 'TETABLE.INC'
      INCLUDE 'INCS:DANS.INC'
      INCLUDE 'INCS:DDCH.INC'
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'INCS:DSOU.INC'
      INCLUDE 'INCS:PSTD.INC'
C
      DATA LASTSA /0/
      DATA LASTSU, LASTID /2*0/
      DATA DISK   /0/
      DATA TLAST /-10./
C-----------------------------------------------------------------------
      IRET = 0
      TR = TIME
C
C     Read file information and allocate LUN for source table if this is
C     the first call (DISK = 0):
C
      IF (DISK .EQ. 0) THEN
         CALL OUVATT (UVFILE, .FALSE., IRET)
         CALL CHECK ('TECCOR', 1, IRET .EQ. 0, IRET)
         CALL OUVCGT (UVFILE, CATBLK, IRET)
         CALL CHECK ('TECCOR', 2, IRET .EQ. 0, IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL FNAGET (UVFILE, 'DISK', TYPE, DIM, IDUM, CDUMMY, IRET)
         DISK = IDUM(1)
         INDISK = DISK
         CALL CHECK ('TECCOR', 3, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                            .AND. (DIM(1) .EQ. 1)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL FNAGET (UVFILE, 'CNO', TYPE, DIM, IDUM, CDUMMY, IRET)
         CNO = IDUM(1)
         INCNO = CNO
         CALL CHECK ('TECCOR', 4, (IRET .EQ. 0) .AND. (TYPE .EQ. OOAINT)
     *                            .AND. (DIM(1) .EQ. 1)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL OBLUN (SULUN, IRET)
         CALL CHECK ('TECCOR', 5, IRET .EQ. 0, IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL UVDGET (UVFILE, 'DATE-OBS', TYPE, DIM, IDUM, DATOBS,
     *                IRET)
         CALL CHECK ('TECCOR', 6, (IRET .EQ. 0) .AND. (TYPE .EQ. OOACAR)
     *                            .AND. (DIM(1) .EQ. 8)
     *                            .AND. (DIM(2) .EQ. 1), IRET)
         IF (IRET .NE. 0) GO TO 999
         CALL JULDAY (DATOBS, JD0)
      END IF
C
C     Make sure array information is up to date:
C
      IF ((IRET .EQ. 0) .AND. (SUBA .NE. LASTSA)) THEN
         CALL GETANT (DISK, CNO, SUBA, CATBLK, ANBUFF, IRET)
         IF (IRET .EQ. 0) THEN
            LASTSA = SUBA
            JD0UTC = JD0 - ANTUTC / (24.0D0 * 3600.0D0)
         ELSE
            WRITE (MSGTXT, 9000) IRET
            CALL MSGWRT (9)
            IRET = 2
            END IF
         END IF
C
C     Make sure source information is up to date:
C     change number if required by missing source number in CL
C
      IF ((IRET.EQ.0) .AND. ((SOURID.NE.LASTSU) .OR. (TR.GT.TLAST)))
     *   THEN
         CALL FNDCOO (0, JD0, SOURID, DISK, CNO, CATBLK, SULUN, TR, DRA,
     *      DDEC, PLANET, IRET)
         IF (IRET.NE.0) THEN
            WRITE (MSGTXT, 9001) IRET, SOURID
            CALL MSGWRT (7)
            LASTID = -SOURID
            IF (IRET.EQ.11) IRET = 0
         ELSE
            LASTID = SOURID
            END IF
         LASTSU = SOURID
         TLAST = TR
         END IF
      IF (IRET.EQ.0) THEN
C
C     Find the antenna in the DANS.INC common blocks
C
         ANTIDX = 0
         DO 10 ANT = 1, NSTNS
            IF (TELNO(ANT).EQ.ANTNO) ANTIDX = ANT
 10         CONTINUE
         IF ((ANTIDX.NE.0) .AND. (LASTID.GT.0)) THEN
            CALL COOELV (ANTIDX, TIME, DRA, DDEC, HA, EL, AZ)
            ZA = (PI / 2.0) - EL
            CALL GETSIP (REAL (STNLAT(ANTIDX)), AZ, ZA, IONALT(),
     *         DLAT, DLON, AZION, ZAION)
            CALL IONTEC (LFACT, REAL (STNLAT(ANTIDX) + DLAT),
     *         REAL (STNLON(ANTIDX) + DLON), JD0UTC + TIME, TEC, IRET)
            IF (IRET.EQ.0) THEN
               DBG = 1.0 / COS (ZAION)
               ALT = IONALT()
               PETROV = (RADIUS / (RADIUS + ALT + TECFAC(2))) *
     *            COS (TECFAC(3) * EL)
               PETROV = TECFAC(1) / SQRT (1.0D0 - PETROV**2)
               TEPATH = PETROV * TEC
C               TEPATH = TECFAC * TEC / COS (ZAION)
               IF (DOIGRF) THEN
                  CALL DATEST (DATOBS, IDATE)
                  CALL DAYNUM (IDATE(1), IDATE(3), IDATE(2), DAYN)
                  DATE = IDATE(1) + DAYN / 365.25D0
                  ALT = IONALT() / 1000.0D0
                  COLAT = 90.0D0 - (STNLAT(ANTIDX)+DLAT) * RAD2DG
                  ELONG = (STNLON(ANTIDX)+DLON) * RAD2DG
                  CALL MGRF14 (0, DATE, 1, ALT, COLAT, ELONG, X, Y, Z,
     *               F)
C                                       nanoTeslas -> gauss
                  B(1) = -Z / 1.D5
                  B(2) = Y / 1.D5
                  B(3) = X / 1.D5
               ELSE
                  CALL MAGDIP (REAL (STNLAT(ANTIDX) + DLAT),
     *               REAL (STNLON(ANTIDX) + DLON),
     *               RADIUS + IONALT(), B)
                  END IF
               MAG = -1.0 * (B(1) * COS (ZAION)
     *                       + B(2) * SIN (ZAION) * SIN (AZION)
     *                       + B(3) * SIN (ZAION) * COS (AZION))
               IFR = 2.6E-17 * MAG * TEPATH
               IF (DODDEL) THEN
                  DO 20 POL = 1, NUMPOL
                     DISP(POL) = 40.28 * TEPATH / (VELITE ** 3)
                     DDISP(POL) = 0.0
   20                CONTINUE
                  END IF
               CALL TETABL ('WRIT', TEVER, TECTYP, TIME, SOURID, ANTIDX,
     *            HA, AZ, ZA, AZION, ZAION, DLON, DLAT, B, TEPATH, MAG,
     *            TEC, IFR, DISP)
            ELSE
               IFR = FBLANK
               IF (DODDEL) THEN
                  DO 30 POL = 1, NUMPOL
                     DISP(POL) = FBLANK
                     DDISP(POL) = FBLANK
   30                CONTINUE
                  END IF
               IRET = 1
               END IF
C                                       No antenna/source data.
         ELSE
            IFR = FBLANK
            IF (DODDEL) THEN
               DO 40 POL = 1, NUMPOL
                  DISP(POL) = FBLANK
                  DDISP(POL) = FBLANK
   40             CONTINUE
               END IF
            IRET = 1
            END IF
         END IF
C
  999 RETURN
C-----------------------------------------------------------------------
 9000 FORMAT ('TECCOR: ERROR',I4,' READING ARRAY INFORMATION')
 9001 FORMAT ('TECCOR: ERROR',I4,' READING SOURCE',I4,' INFORMATION')
      END
      SUBROUTINE TETABL (OPCODE, TEVER, TECTYP, TIME, SOURCE, ANTEN,
     *   RHA, RAZ, RZA, RAZION, RZAION, RDLON, RDLAT, B, TEPATH, MAG,
     *   TEC, IFR, DISP)
C-----------------------------------------------------------------------
C   opens, write, closes TE table
C   INPUTS:
C      OPCODE   C*4    'WRIT', 'CLOS'
C      TECTYP   C*8    TEC data type
C      TIME     D      time days
C      SOURID   I      source number
C      ANTIDX   I      antenna number
C      HA       R      hour angle rad
C      AZ       R      azimuth rad
C      EL       R      elevation rad
C      AZION    R      pierce point azimuth rad
C      ZAION    R      pierce point zenith angle rad
C      DLON     R      longitude offset rad
C      DLAT     R      latitude offset rad
C      B        R(3)   mag field vector
C      TEPATH   R      TEC path length
C      MAG      R      projected mag field
C      TEC      R      totsl TEC
C      IFR      R      ionospheric rotation measure rad/m/m
C      DISP     R      dispersive delay
C   Output:
C      TEVER    I      TE table version on file create
C-----------------------------------------------------------------------
      CHARACTER OPCODE*4, TECTYP*8
      DOUBLE PRECISION TIME
      INTEGER   SOURCE, TEVER, ANTEN
      REAL      RHA, RAZ, RZA, RAZION, RZAION, RDLON, RDLAT, B(3),
     *   TEPATH, MAG, TEC, IFR, DISP(2)
C
      INCLUDE 'INCS:PUVD.INC'
      INCLUDE 'INCS:PSTD.INC'
      INCLUDE 'TETABLE.INC'
      INCLUDE 'INCS:DANT.INC'
      INCLUDE 'INCS:DCAT.INC'
      INCLUDE 'INCS:DMSG.INC'
      INTEGER   LUN, IERR
      REAL      HA, AZ, ZA, AZION, ZAION, DLON, DLAT, TEP, STEC
C-----------------------------------------------------------------------
      IF (DOTABL) THEN
C                                       close
         IF (OPCODE.EQ.'CLOS') THEN
            CALL TABTE (OPCODE, TEBUFF, ITERNO, TEKOLS, TENUMV, TIME,
     *         SOURCE, ANTEN, HA, AZ, ZA, AZION, ZAION, DLON, DLAT, B,
     *         TEPATH, MAG, TEC, IFR, DISP, IERR)
            IF (IERR.NE.0) THEN
               WRITE (MSGTXT,1000) IERR, 'CLOSING TE TABLE'
               GO TO 990
               END IF
         ELSE IF (OPCODE.EQ.'WRIT') THEN
C                                       create file
            IF (TENUMV(1).LE.0) THEN
               TEVER = 0
               LUN = 83
               CALL TEINI ('WRIT', TEBUFF, INDISK, INCNO, TEVER, CATBLK,
     *            LUN, ITERNO, TEKOLS, TENUMV, RDATE, TECTYP, IERR)
               IF (IERR.NE.0) THEN
                  WRITE (MSGTXT,1000) IERR, 'CREATING TE TABLE'
                  GO TO 990
                  END IF
               END IF
C                                       convert to degrees
            HA = RHA * RAD2DG
            AZ = RAZ * RAD2DG
            ZA = RZA * RAD2DG
            AZION = RAZION * RAD2DG
            ZAION = RZAION * RAD2DG
            DLON = RDLON * RAD2DG
            DLAT = RDLAT * RAD2DG
            TEP = TEPATH / 1.0E16
            STEC = TEC / 1.0E16
C                                       write table row
            CALL TABTE (OPCODE, TEBUFF, ITERNO, TEKOLS, TENUMV, TIME,
     *         SOURCE, ANTEN, HA, AZ, ZA, AZION, ZAION, DLON, DLAT, B,
     *         TEP, MAG, STEC, IFR, DISP, IERR)
            IF (IERR.NE.0) THEN
               WRITE (MSGTXT,1000) IERR, 'WRITING TE TABLE'
               GO TO 990
               END IF
            END IF
         END IF
      GO TO 999
C
 990  CALL MSGWRT (8)
      DOTABL = .FALSE.
      MSGTXT = 'TE TABLE WILL NOT BE WRITTEN'
      CALL MSGWRT (8)
C
 999  RETURN
C-----------------------------------------------------------------------
 1000 FORMAT ('TETABL ERROR',I4,' ON ',A)
      END
      SUBROUTINE TECHIS (OCLTAB, INPUTS, DODDEL, DOIGRF, TECFAC, LFACT,
     *   OUTVER, TEVER, NUMMOD, IRET)
C-----------------------------------------------------------------------
C   Update history file.
C
C   Inputs:
C      OCLTAB  C*(*)   Name of TABLE object used to access output CL
C                       table
C      INPUTS  C*(*)   Name of inputs object used to access adverbs
C      DODDEL  L       Were dispersive delays calculated?
C      OUTVER  I       Version number of output CL table
C      TEVER   I       Version number of output TE table
C      NUMMOD  I       Number of modified CL records
C      DOIGRF  L       Is the IGRF v13 model to be used
C      TECFAC  R       Scale TEC factor
C
C   Outputs:
C      IRET    I       Return status:
C                       0 - history updated
C                       1 - I/O error
C
C   Preconditions:
C      OCLTAB is initialized
C      INPUTS is initialized
C      OUTVER > 0
C      NUMMOD >= 0
C-----------------------------------------------------------------------
      CHARACTER OCLTAB*(*), INPUTS*(*)
      LOGICAL   DODDEL, DOIGRF
      REAL      LFACT, TECFAC(3)
      INTEGER   OUTVER, TEVER, NUMMOD, IRET
C
C     Local variables
C
C     NLIST   Number of adverbs to list
C     LIST    Adverbs to list
C     ENTRY   History file entry
C
      INTEGER   NLIST
      PARAMETER (NLIST = 11)
      CHARACTER LIST(NLIST)*8, ENTRY*72
C
      INCLUDE 'INCS:DMSG.INC'
C
      DATA LIST /'INNAME  ', 'INCLASS ', 'INSEQ   ', 'INDISK  ',
     *           'INFILE  ', 'NFILES  ', 'SUBARR  ', 'ANTENNAS',
     *           'GAINVER ', 'GAINUSE ', 'APARM   ' /
C-----------------------------------------------------------------------
      CALL OHTIME (OCLTAB, IRET)
      IF (IRET .EQ. 0) THEN
         CALL OHLIST (INPUTS, LIST, NLIST, OCLTAB, IRET)
         IF (IRET .EQ. 0) THEN
            WRITE (ENTRY, 1000) OUTVER
            CALL OHWRIT (ENTRY, OCLTAB, IRET)
            IF (IRET .EQ. 0) THEN
               WRITE (ENTRY, 1001) NUMMOD
               CALL OHWRIT (ENTRY, OCLTAB, IRET)
               IF ((IRET.EQ.0) .AND. DODDEL) THEN
                  ENTRY = '/ Calculated dispersive delay'
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  END IF
               IF (IRET.EQ.0) THEN
                  WRITE (ENTRY,4000) LFACT
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  END IF
               IF (IRET.EQ.0) THEN
                  ENTRY = '/ Used older model of Earth magnetic field'
                  IF (DOIGRF) ENTRY =
     *               '/ Used IGRF model 13 of Earth magnetic field'
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  END IF
               IF (IRET.EQ.0) THEN
                  WRITE (ENTRY,4001) TECFAC(1)
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  WRITE (ENTRY,4002) TECFAC(2) / 1000.0
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  WRITE (ENTRY,4003) TECFAC(3)
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
                  WRITE (ENTRY,4004) TEVER
                  CALL OHWRIT (ENTRY, OCLTAB, IRET)
               END IF
            END IF
         END IF
      END IF
C
      IF (IRET .NE. 0) THEN
         WRITE (MSGTXT, 9000) IRET
         CALL MSGWRT (9)
         IRET = 1
      END IF
C-----------------------------------------------------------------------
 1000 FORMAT ('Created CL table version ', I4)
 1001 FORMAT ('Updated ', I6, ' CL table rows')
 4000 FORMAT ('LFACT    =',F8.5,5X,'/ Follow ionosphere fraction')
 4001 FORMAT ('TECFACT  =',F8.5,5X,'/ Scale TEC factor')
 4002 FORMAT ('DELTAEL  =',F8.2,5X,'/ elevation offset km')
 4003 FORMAT ('SCALEEL  =',F8.5,5X,'/ Scale elevation')
 4004 FORMAT ('TEVERS   =',I5,8X,'/ TE table version written')
 9000 FORMAT ('TECHIS: ERROR ', I4, ' UPDATING HISTORY FILE')
      END
      SUBROUTINE TECVAL (MAP, LONDIM, NLONG, NLAT, MINLON, DLONG,
     *                   MINLAT, DLAT, LONG, LAT, TEC, IRET)
C-----------------------------------------------------------------------
C   Find the zenith TEC at LAT and LONG.
C
C   Inputs:
C      MAP        R(LONDIM, *)     Map of zenith TEC; actual map values
C                                    are stored in the subrange
C                                    MAP(1:NLONG, 1:NLAT) and cells for
C                                    which the zenith TEC is not
C                                    available are set to FBLANK.
C      LONDIM     I                Leading dimension (longitude) of MAP
C                                    array
C      NLONG      I                Number of longitude cells in MAP
C      NLAT       I                Number of longitude cells in MAP
C      MINLON     R                East longitude at MAP(1, :) in
C                                    degrees
C      DLONG      R                Spacing in longitude
C      MINLAT     R                Latitude at MAP(:, 1) in degrees
C      DLAT       R                Spacing in latitude
C      LAT        R                Latitude at which zenith TEC is
C                                    desired in degrees
C      LONG       R                East longitude at which zenith TEC
C                                    is desired in degrees
C
C   Outputs:
C      TEC        R                Zenith TEC in electrons per square
C                                    metre
C      IRET       I                Return status:
C                                    0 - TEC available
C                                    1 - TEC data missing
C
C   Preconditions:
C      LONDIM >= NLONG >= 2
C      NLAT  >= 2
C      DLONG > 0.0
C      DLAT > 0.0
C      -180.0 <= LONG <= +180.0
C      -90.0 <= LAT <= +90.0
C
C   Postconditions:
C      IRET = 0 implies TEC /= FBLANK and TEC > 0.0
C      IRET = 1 implies TEC = FBLANK
C-----------------------------------------------------------------------
      INTEGER   LONDIM
      REAL      MAP(LONDIM, *)
      INTEGER   NLONG, NLAT
      REAL      MINLON, DLONG, MINLAT, DLAT, LAT, LONG, TEC
      INTEGER   IRET
C
C     Local variables
C
C     I     Coordinate of cell with next lower longitude to LONG in MAP
C     J     Coordinate of cell with next lower latitude to LAT in MAP
C     WX    Interpolation weight for x-axis (longitude)
C     WY    Interpolation weight for y-axis (latitude)
C
      INTEGER   I, J
      REAL      WX, WY
C
      INCLUDE 'INCS:DDCH.INC'
C-----------------------------------------------------------------------
      I = INT ((LONG - MINLON) / DLONG) + 1
      J = INT ((LAT - MINLAT) / DLAT) + 1
      IF ((1 .LE. I) .AND. (I .LT. NLONG) .AND. (1 .LE. J)
     *     .AND. (J .LT. NLAT)) THEN
         WX = (LONG - MINLON - (I - 1) * DLONG) / DLONG
         WY = (LAT - MINLAT - (J - 1) * DLAT) / DLAT
         IF ((MAP(I, J) .NE. FBLANK) .AND. (MAP(I, J + 1) .NE. FBLANK)
     *       .AND. (MAP(I + 1, J) .NE. FBLANK)
     *       .AND. (MAP(I + 1, J + 1) .NE. FBLANK)) THEN
            TEC = (1.0 - WX) * (1.0 - WY) * MAP(I, J)
     *            + (1.0 - WX) * WY * MAP(I, J + 1)
     *            + WX * (1.0 - WY) * MAP(I + 1, J)
     *            + WX * WY * MAP(I + 1, J + 1)
            IRET = 0
         ELSE
            TEC = FBLANK
            IRET = 1
         END IF
      ELSE
         TEC = FBLANK
         IRET = 1
      END IF
      END
LOCAL INCLUDE 'IONEX.INC'
C
C   Private structures maintaining information from IONEX maps
C
C   MAXMAP   Maximum number of TEC maps
C   MAXLON   Maximum number of longitude cells
C   MAXLAT   Maximum number of latitude cells
C   NMAPS    Number of TEC maps expected in one file
C   NMAPRD   Number of TEC maps read in one file
C   NLONG    Number of longitude cells in maps
C   NLAT     Number of latitude cells in maps
C   MINLON   Smallest East longitude in maps in degrees
C   MINLAT   Smallest latitude in degrees
C   DLONG    Longitude spacing in degrees
C   DLAT     Latitude spacing in degrees
C   MAPTIM   Julian day number for each map
C   MAPS     TEC maps
C   ALT      Height of ionosphere
C   EXPONT   Exponent for IONEX file
C   TOTMAP   Total number of maps read in all files
C
      INTEGER   MAXMAP, MAXLON, MAXLAT
      PARAMETER (MAXMAP = 300)
      PARAMETER (MAXLON = 100)
      PARAMETER (MAXLAT = 80)
      INTEGER   NMAPS, NMAPRD, NLONG, NLAT, TOTMAP
      REAL      MINLON, MINLAT, DLONG, DLAT
      DOUBLE PRECISION MAPTIM(MAXMAP)
      REAL      MAPS(MAXLON, MAXLAT, MAXMAP), ALT
      INTEGER   EXPONT
C
      COMMON /IONEX/ MAPTIM, MAPS, NMAPS, NMAPRD, NLONG, NLAT, MINLON,
     *               MINLAT, DLONG, DLAT, ALT, EXPONT, TOTMAP
      SAVE /IONEX/
C
LOCAL END
      SUBROUTINE IONINI (FNAME, NFILE, IRET)
C-----------------------------------------------------------------------
C   Initialize IONEX module and parse input file.
C
C   Inputs:
C      FNAME    C*(*)   Input file name
C      NFILE    I       File number, if multiple IONEX files requested
C
C   Outputs:
C      IRET     I       Return status
C                          0 - module initialized
C                          1 - failed to read maps
C                          2 - failed to read header
C                          3 - can not open file
C-----------------------------------------------------------------------
      CHARACTER FNAME*(*)
      INTEGER   IRET, NFILE
C
C     Local variables
C
C     TXTLUN   AIPS LUN for input file
C     FIND     FTAB index for input file
C     IRET2    Disposable return status
C
      INTEGER   TXTLUN
      PARAMETER (TXTLUN = 10)
      INTEGER   FIND, IRET2
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      CALL ZTXOPN ('READ', TXTLUN, FIND, FNAME, .FALSE., IRET)
      IF (IRET .EQ. 0) THEN
         CALL IONHDR (TXTLUN, FIND, IRET)
         IF (IRET .EQ. 0) THEN
            IF (NFILE .EQ. 1) NMAPRD = 0
C                                       NMAPS is the number of TEC
C                                       maps in the IONEX file
C
C           Invariant: IRET = 0 implies that NMAPRD TEC maps have been
C                               read from the input file
C           Bound: NMAPS+TOTMAP - NMAPRD
C
   10       IF ((IRET .EQ. 0) .AND. (NMAPRD .NE. NMAPS+TOTMAP)) THEN
               CALL IONMAP (TXTLUN, FIND, IRET)
               IF (IRET .NE. 0) THEN
                  WRITE (MSGTXT, 9010) IRET
                  CALL MSGWRT (9)
                  IRET = 1
               END IF
               GO TO 10
            END IF
         ELSE
            WRITE (MSGTXT, 9011) IRET
            CALL MSGWRT (9)
            IRET = 2
         END IF
C
C           Close input file; ignore errors at this stage
C
         CALL ZTXCLS (TXTLUN, FIND, IRET2)
         TOTMAP=NMAPRD
      ELSE
         WRITE (MSGTXT, 9012) IRET
         CALL MSGWRT (9)
         IRET = 3
      END IF
C-----------------------------------------------------------------------
 9010 FORMAT ('IONINI: ERROR ', I4, ' READING IONEX DATA')
 9011 FORMAT ('IONINI: ERROR ', I4, ' READING IONEX HEADER')
 9012 FORMAT ('IONINI: ERROR ', I4, ' OPENING INPUT TEXT FILE')
      END
      SUBROUTINE IONHDR (LUN, FIND, IRET)
C-----------------------------------------------------------------------
C   Parse the header of an IONEX file that is open for reading on LUN
C   with FTAB index FIND.
C
C   Inputs:
C      LUN      AIPS LUN for input file
C      FIND     FTAB index of input file
C
C   Output:
C      IRET     Return status
C                  0 - header parsed and file is usable
C                  1 - file unusable
C                  2 - wrong IONEX version or data type
C                  3 - input file is not an IONEX file
C                  4 - header data missing
C                 10 - I/O error detected
C-----------------------------------------------------------------------
      INTEGER   LUN, FIND, IRET
C
C     Local variables
C
C     LINE    Line buffer
C     VERS    IONEX revision
C     TYPE    Data type code
C     SYSTEM  Observing system
C
      CHARACTER LINE*80
      REAL      VERS
      CHARACTER TYPE*1, SYSTEM*3
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
C
C     Set default exponent
C
      EXPONT = -1
C
      NMAPS = 0
      NLONG = 0
      NLAT = 0
      ALT = 0.0
      CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
      IF (IRET .EQ. 0) THEN
         IF (LINE(61:80) .EQ. 'IONEX VERSION / TYPE') THEN
            READ (LINE, FMT = '(F8.1, 12X, A1, 19X, A3, 37X)',
     *            IOSTAT = IRET) VERS, TYPE, SYSTEM
            IF (IRET .EQ. 0) THEN
               IF ((VERS .EQ. 1.0) .AND. (TYPE .EQ. 'I')) THEN
C
C                 Read each header record
C
   10             IF ((IRET .EQ. 0)
     *                .AND. (LINE(61:80) .NE. 'END OF HEADER       '))
     *               THEN
                     CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
                     IF (IRET .EQ. 0) THEN
                        IF (LINE(61:80)
     *                      .EQ. '# OF MAPS IN FILE   ') THEN
                           CALL IONNMP (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9010) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        ELSE IF (LINE(61:80)
     *                           .EQ. 'MAP DIMENSION       ') THEN
                           CALL IONDIM (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9011) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        ELSE IF (LINE(61:80)
     *                           .EQ. 'HGT1 / HGT2 / DHGT ') THEN
                           CALL IONHGT (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9012) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        ELSE IF (LINE(61:80)
     *                           .EQ. 'LAT1 / LAT2 / DLAT ') THEN
                           CALL IONLAT (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9013) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        ELSE IF (LINE(61:80)
     *                           .EQ. 'LON1 / LON2 / DLON ') THEN
                           CALL IONLON (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9014) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        ELSE IF (LINE(61:80)
     *                           .EQ. 'EXPONENT           ') THEN
                           CALL IONEXP (LINE, IRET)
                           IF (IRET .NE. 0) THEN
                              WRITE (MSGTXT, 9015) IRET
                              CALL MSGWRT (9)
                              IRET = 1
                           END IF
                        END IF
                     ELSE
                        WRITE (MSGTXT, 9016) IRET
                        CALL MSGWRT (9)
                        IRET = 10
                     END IF
                     GO TO 10
                  END IF
               ELSE
                  WRITE (MSGTXT, 9017) VERS, TYPE
                  CALL MSGWRT (9)
                  IRET = 2
               END IF
            ELSE
               WRITE (MSGTXT, 9018)
               CALL MSGWRT (9)
               IRET = 10
            END IF
         ELSE
            WRITE (MSGTXT, 9019)
            CALL MSGWRT (9)
            IRET = 3
         END IF
      ELSE
         WRITE (MSGTXT, 9016) IRET
         CALL MSGWRT (9)
         IRET = 10
      END IF
C
C     Check that nothing important was missing:
C
      IF (IRET .EQ. 0) THEN
         IF (NMAPS .EQ. 0) THEN
            WRITE (MSGTXT, 9020)
            CALL MSGWRT (9)
            IRET = 4
         END IF
         IF (NLAT .EQ. 0) THEN
            WRITE (MSGTXT, 9021)
            CALL MSGWRT (9)
            IRET = 4
         END IF
         IF (NLONG .EQ. 0) THEN
            WRITE (MSGTXT, 9022)
            CALL MSGWRT (9)
            IRET = 4
         END IF
         IF (ALT .EQ. 0.0) THEN
            WRITE (MSGTXT, 9023)
            CALL MSGWRT (9)
            IRET = 4
         END IF
      END IF
C-----------------------------------------------------------------------
 9010 FORMAT ('IONHDR: ERROR ', I4, ' READING NUMBER OF MAPS')
 9011 FORMAT ('IONHDR: ERROR ', I4, ' READING MAP DIMENSIONS')
 9012 FORMAT ('IONHDR: ERROR ', I4, ' READING HEIGHT CARD')
 9013 FORMAT ('IONHDR: ERROR ', I4, ' READING LATITUDE CARD')
 9014 FORMAT ('IONHDR: ERROR ', I4, ' READING LONGITUDE CARD')
 9015 FORMAT ('IONHDR: ERROR ', I4, ' READING EXPONENT CARD')
 9016 FORMAT ('IONHDR: ERROR ', I4, ' READING TEXT FILE')
 9017 FORMAT ('IONHDR: UNKNOWN VERSION (', F6.1, ') OR TYPE (''', A1,
     *        ''')')
 9018 FORMAT ('IONHDR: ERROR PARSING FIRST RECORD')
 9019 FORMAT ('IONHDR: INPUT FILE IS NOT IN IONEX FORMAT')
 9020 FORMAT ('IONHDR: NUMBER OF MAPS IS MISSING')
 9021 FORMAT ('IONHDR: LATITUDE SPECIFICATION IS MISSING')
 9022 FORMAT ('IONHDR: LONGITUDE SPECIFICATION IS MISSING')
 9023 FORMAT ('IONHDR: HEIGHT SPECIFICATION IS MISSING')
      END
      SUBROUTINE IONNMP (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse a number of maps record in LINE.
C
C   Input:
C      LINE     C*80     Number of maps record
C
C   Output:
C      IRET     I        Return status:
C                          0 - valid number of maps
C                          1 - invalid number of maps
C                          2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(I6, 74X)', IOSTAT = IRET) NMAPS
      IF (IRET .EQ. 0) THEN
         IF ((1 .LE. NMAPS) .AND. (NMAPS .LE. MAXMAP)) THEN
            IRET = 0
         ELSE IF (NMAPS .LT. 1) THEN
            WRITE (MSGTXT, 9000) NMAPS
            CALL MSGWRT (9)
            IRET = 1
         ELSE IF (NMAPS .GT. MAXMAP) THEN
            WRITE (MSGTXT, 9001) NMAPS, MAXMAP
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9002)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONNMP: INVALID NUMBER OF MAPS ', I6)
 9001 FORMAT ('IONNMP: TOO MANY MAPS - ', I6 , ' EXCEEDS MAX ', I6)
 9002 FORMAT ('IONNMP: FAILED TO PARSE NUMBER OF MAPS')
      END
      SUBROUTINE IONDIM (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse dimensions record
C
C   Input
C      LINE    C*80    Dimensions record
C
C   Output
C      IRET    I       Return status
C                        0 - dimensions valid
C                        1 - dimensions invalid
C                        2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
C     Local variables
C
C     DIM      Dimensionality of maps
C
      INTEGER   DIM
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(I6, 74X)', IOSTAT = IRET) DIM
      IF (IRET .EQ. 0) THEN
         IF (DIM .EQ. 2) THEN
            IRET = 0
         ELSE
            WRITE (MSGTXT, 9000) DIM
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9001)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONDIM: CAN NOT USE ', I6, ' DIMENSIONAL MAPS')
 9001 FORMAT ('IONDIM: ERROR PARSING NUMBER OF DIMENSIONS')
      END
      SUBROUTINE IONHGT (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse height record in LINE
C
C   Input:
C      LINE   C*80    Height record
C
C   Output:
C      IRET   I       Return status
C                       0 - record parsed
C                       1 - not constant height data
C                       2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
C     Local variables
C
C     HGT1    Minimum height
C     HGT2    Maximum height
C     DHGT    Height grid spacing
C
      REAL      HGT1, HGT2, DHGT
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(2X, 3F6.1, 60X)', IOSTAT = IRET) HGT1, HGT2,
     *                                                     DHGT
      IF (IRET .EQ. 0) THEN
         IF (HGT1 .EQ. HGT2) THEN
            ALT = 1000.0 * HGT1
            IRET = 0
         ELSE
            WRITE (MSGTXT, 9000)
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9001)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONHGT: CAN ONLY HANDLE CONSTANT HEIGHT DATA')
 9001 FORMAT ('IONHGT: ERROR PARSING HEIGHT RECORD')
      END
      SUBROUTINE IONLAT (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse latitude record in LINE
C
C   Input:
C      LINE   C*80    Latitude record
C
C   Output:
C      IRET   I       Return status
C                       0 - record parsed
C                       1 - invalid number of latitudes
C                       2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
C     Local variables
C
C     LAT1    Minimum latitude
C     LAT2    Maximum latitude
C
      REAL      LAT1, LAT2
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(2X, 3F6.1, 60X)', IOSTAT = IRET) LAT1, LAT2,
     *                                                     DLAT
      IF (IRET .EQ. 0) THEN
         NLAT = INT (ABS(LAT2 - LAT1) / ABS (DLAT)) + 1
         IF ((1 .LE. NLAT) .AND. (NLAT .LE. MAXLAT)) THEN
            DLAT = ABS (DLAT)
            MINLAT = MIN (LAT1, LAT2)
         ELSE IF (NLAT .LT. 1) THEN
            WRITE (MSGTXT, 9000) NLAT
            CALL MSGWRT (9)
            IRET = 1
         ELSE IF (NLAT .GT. MAXLAT) THEN
            WRITE (MSGTXT, 9001) NLAT, MAXLAT
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9002)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONLAT: INVALID NUMBER OF LATITUDES ', I6)
 9001 FORMAT ('IONLAT: TOO MANY LATITUDES - ', I6, ' EXCEEDS ', I6)
 9002 FORMAT ('IONLAT: ERROR PARSING LATITUDE RECORD')
      END
      SUBROUTINE IONLON (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse longitude record in LINE
C
C   Input:
C      LINE   C*80    Longitude record
C
C   Output:
C      IRET   I       Return status
C                       0 - record parsed
C                       1 - invalid number of longitudes
C                       2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
C     Local variables
C
C     LON1    Minimum longitude
C     LON2    Maximum longitude
C
      REAL      LON1, LON2
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(2X, 3F6.1, 60X)', IOSTAT = IRET) LON1, LON2,
     *                                                     DLONG
      IF (IRET .EQ. 0) THEN
         NLONG = INT (ABS(LON2 - LON1) / ABS (DLONG)) + 1
         IF ((1 .LE. NLONG) .AND. (NLONG .LE. MAXLON)) THEN
            DLONG = ABS (DLONG)
            MINLON = MIN (LON1, LON2)
         ELSE IF (NLONG .LT. 1) THEN
            WRITE (MSGTXT, 9000) NLONG
            CALL MSGWRT (9)
            IRET = 1
         ELSE IF (NLONG .GT. MAXLON) THEN
            WRITE (MSGTXT, 9001) NLONG, MAXLON
            CALL MSGWRT (9)
            IRET = 1
         END IF
      ELSE
         WRITE (MSGTXT, 9002)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONLON: INVALID NUMBER OF LONGITUDES ', I6)
 9001 FORMAT ('IONLON: TOO MANY LONGITUDES - ', I6, ' EXCEEDS ', I6)
 9002 FORMAT ('IONLON: ERROR PARSING LONGITUDE RECORD')
      END
      SUBROUTINE IONEXP (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse an exponent record in LINE.
C
C   Inputs:
C      LINE    C*80     Exponent record
C
C   Output:
C      IRET    I        Return status
C                         0 - parsed exponent
C                         1 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER IRET
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(I6, 74X)', IOSTAT = IRET) EXPONT
      IF (IRET .NE. 0) THEN
         WRITE (MSGTXT, 9000)
         CALL MSGWRT (9)
         IRET = 1
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONEXP: ERROR PARSING EXPONENT RECORD')
      END
      SUBROUTINE IONMAP (LUN, FIND, IRET)
C-----------------------------------------------------------------------
C   Parse the next TEC map in an IONEX file open for reading on LUN.
C   This should be called after IONHDR.
C
C   Inputs:
C      LUN     I       LUN on which file is open
C      FIND    I       FTAB index of file
C
C   Output:
C      IRET    I       Return status
C                        0 - map read
C                        1 - bad epoch
C                        2 - bad data block
C                        3 - epoch out of order
C                        4 - bad exponent
C                       10 - I/O error
C-----------------------------------------------------------------------
      INTEGER   LUN, FIND, IRET, JRET
C
C     Local variables
C
C     LINE      Line buffer
C
      CHARACTER LINE*80
C
      INCLUDE 'INCS:DDCH.INC'
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
      JRET = -1
C
C     Skip to start of TEC map, updating the current exponent if any
C     exponent records are found:
C
   10 IF ((IRET .EQ. 0)
     *    .AND. (LINE(61:80) .NE. 'START OF TEC MAP    ')) THEN
         IF (LINE(61:80) .EQ. 'EXPONENT            ') THEN
            CALL IONEXP (LINE, IRET)
            IF (IRET .EQ. 0) THEN
               CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
            ELSE
               WRITE (MSGTXT, 9010) IRET
               CALL MSGWRT (9)
               IRET = 4
            END IF
         ELSE
            CALL ZTXIO  ('READ', LUN, FIND, LINE, IRET)
         END IF
         GO TO 10
      END IF
      IF (IRET .EQ. 0) THEN
C
C        Must be the start of a TEC map.
C
         NMAPRD = NMAPRD + 1
         MAPTIM(NMAPRD) = 0.0D0
         CALL RFILL (MAXLON * MAXLAT, FBLANK, MAPS(1, 1, NMAPRD))
C
C        Parse map records:
C
   20    IF ((IRET .EQ. 0)
     *      .AND. (LINE(61:80) .NE. 'END OF TEC MAP    ')) THEN
C
C           Skip to next important record:
C
   30       IF ((IRET .EQ. 0)
     *          .AND. (LINE(61:80) .NE. 'EXPONENT            ')
     *          .AND. (LINE(61:80) .NE. 'EPOCH OF CURRENT MAP')
     *          .AND. (LINE(61:80) .NE. 'LAT/LON1/LON2/DLON/H')
     *          .AND. (LINE(61:80) .NE. 'END OF TEC MAP      ')) THEN
               CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
               GO TO 30
            END IF
            IF (IRET .EQ. 0) THEN
C
C              Parse the record:
C
               IF (LINE(61:80) .EQ. 'EXPONENT            ') THEN
                  CALL IONEXP (LINE, IRET)
                  IF (IRET .NE. 0) THEN
                     WRITE (MSGTXT, 9010) IRET
                     CALL MSGWRT (9)
                     IRET = 4
                  END IF
               ELSE IF (LINE(61:80) .EQ. 'EPOCH OF CURRENT MAP') THEN
                  CALL IONEPO (LINE, IRET)
                  IF (IRET .NE. 0) THEN
                     WRITE (MSGTXT, 9030) IRET
                     CALL MSGWRT (9)
                     IRET = 1
                  END IF
                  IF (MAPTIM(NMAPRD) .EQ. MAPTIM(NMAPRD - 1) .AND.
     *               (MAPTIM(NMAPRD)-AINT(MAPTIM(NMAPRD)).EQ.0.5)) THEN
C
C              The same time as the last record of the last file.
C              Over-write the old one, this is more likely correct.
C
                     NMAPRD = NMAPRD - 1
                     NMAPS = NMAPS - 1
                     JRET = 1
C                    GO TO 40
                  END IF
               ELSE IF (LINE(61:80) .EQ. 'LAT/LON1/LON2/DLON/H') THEN
                  CALL IONBLK (LINE, LUN, FIND, IRET)
                  IF (IRET .NE. 0) THEN
                     WRITE (MSGTXT, 9031) IRET
                     CALL MSGWRT (9)
                     IRET = 2
                  END IF
               END IF
            ELSE
               WRITE (MSGTXT, 9032) IRET
               CALL MSGWRT (9)
               IRET = 10
            END IF
C
C           Advance over parsed record
C
            CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
            IF (IRET .NE. 0) THEN
               WRITE (MSGTXT, 9032) IRET
               CALL MSGWRT (9)
               IRET = 10
            END IF
            GO TO 20
         END IF
C
C        Check for times out of order:
C
         IF ((IRET .EQ. 0) .AND. (NMAPRD .GT. 1)) THEN
            IF (MAPTIM(NMAPRD) .LE. MAPTIM(NMAPRD - 1)) THEN
               WRITE (MSGTXT, 9033)
               CALL MSGWRT (9)
               IRET = 3
            ENDIF
         END IF
      ELSE
         WRITE (MSGTXT, 9032) IRET
         CALL MSGWRT (9)
         END IF
      IF (JRET .GT. 0) THEN
         WRITE(MSGTXT, 9034)
         CALL MSGWRT (4)
         END IF
C-----------------------------------------------------------------------
 9010 FORMAT ('IONMAP: ERROR ', I4, ' READING EXPONENT')
 9030 FORMAT ('IONMAP: ERROR ', I4, ' READING EPOCH')
 9031 FORMAT ('IONMAP: ERROR ', I4, ' READING DATA BLOCK')
 9032 FORMAT ('IONMAP: ERROR ', I4, ' READING INPUT FILE')
 9033 FORMAT ('IONMAP: FAULTY INPUT FILE - TIMES OUT OF ORDER')
 9034 FORMAT ('IONMAP: last TEC map of previous file over-written',
     *   ' this is good')
      END
      SUBROUTINE IONEPO (LINE, IRET)
C-----------------------------------------------------------------------
C   Parse an IONEX map epoch record in LINE.
C
C   Input:
C      LINE   C*80      Map epoch record
C
C   Output:
C      IRET   I         Return status
C                         0 - epoch read
C                         1 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   IRET
C
C     Local variables:
C
C     IT     Broken-down date and time
C
      INTEGER   IT(6)
C
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(6I6, 44X)', IOSTAT = IRET) IT
      IF (IRET .EQ. 0) THEN
         CALL DAT2JD (IT, MAPTIM(NMAPRD))
      ELSE
         WRITE (MSGTXT, 9000)
         CALL MSGWRT (9)
         IRET = 1
      END IF
C-----------------------------------------------------------------------
 9000 FORMAT ('IONEPO: ERROR PARSING MAP EPOCH RECORD')
      END
      SUBROUTINE IONBLK (LINE, LUN, FIND, IRET)
C-----------------------------------------------------------------------
C   Read a data block from an IONEX file open on LUN and positioned
C   at the line following the 'LAT/LON1/LON2/DLON' record which should
C   be stored in LINE.
C
C   Inputs:
C      LINE    C*80     Header for the data block
C      LUN     I        Logical unit number for input file
C      FIND    I        FTAB index of input file
C
C   Output:
C      IRET    I        Return status
C                         0 - data read
C                         1 - error reading file
C                         2 - parse error
C-----------------------------------------------------------------------
      CHARACTER LINE*80
      INTEGER   LUN, FIND, IRET
C
C     Local variables
C
C     LAT      Latitude of data block (degrees)
C     LATIDX   Index of latitude in map
C     LON1     First longitude in data block (degrees)
C     LON2     Last longitude in data block (degrees)
C     DL       Longitude spacing in degrees
C     LON      Longitude of current data point
C     LONIDX   Index of longitude in map
C     H        Height of map
C     COUNT    Number of values left to read
C     NREAD    Number of values left on current line
C     OFF      Character offset in current line
C     VAL      Current value
C
      REAL     LAT, LON1, LON2, DL, LON, H
      INTEGER  LATIDX, LONIDX, COUNT, NREAD, OFF, VAL
C
      INCLUDE 'INCS:DDCH.INC'
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      READ (LINE, FMT = '(2X, 5F6.1, 48X)', IOSTAT = IRET) LAT, LON1,
     *                                                     LON2, DL, H
      IF (IRET .EQ. 0) THEN
         LATIDX = NINT ((LAT - MINLAT) / DLAT) + 1
         LON = LON1
         COUNT = NINT ((LON2 - LON1) / DL) + 1
   10    IF ((IRET .EQ. 0) .AND. (COUNT .NE. 0)) THEN
            CALL ZTXIO ('READ', LUN, FIND, LINE, IRET)
            IF (IRET .EQ. 0) THEN
               NREAD = MIN (16, COUNT)
               OFF = 1
   20          IF ((IRET .EQ. 0) .AND. (NREAD .NE. 0)) THEN
                  READ (LINE(OFF:OFF+4), FMT = '(I5)', IOSTAT = IRET)
     *               VAL
                  IF (IRET .EQ. 0) THEN
                     LONIDX = NINT ((LON - MINLON) / DLONG) + 1
                     IF (VAL .EQ. 9999) THEN
                        MAPS(LONIDX, LATIDX, NMAPRD) = FBLANK
                     ELSE
                        MAPS(LONIDX, LATIDX, NMAPRD) = VAL
     *                     * 10.0 ** (16 + EXPONT)
                     END IF
                  ELSE
                     WRITE (MSGTXT, 9020)
                     CALL MSGWRT (9)
                     IRET = 2
                  END IF
                  OFF = OFF + 5
                  LON = LON + DL
                  NREAD = NREAD - 1
                  COUNT = COUNT - 1
                  GO TO 20
               END IF
            ELSE
               WRITE (MSGTXT, 9021) IRET
               CALL MSGWRT (9)
               IRET =1
            END IF
            GO TO 10
         END IF
      ELSE
         WRITE (MSGTXT, 9022)
         CALL MSGWRT (9)
         IRET = 2
      END IF
C-----------------------------------------------------------------------
 9020 FORMAT ('IONBLK: ERROR PARSING DATA VALUE')
 9021 FORMAT ('IONBLK: ERROR ', I4, ' READING INPUT FILE')
 9022 FORMAT ('IONBLK: ERROR PARSING DATA BLOCK HEADER')
      END
      DOUBLE PRECISION FUNCTION IONFIR()
C-----------------------------------------------------------------------
C   Earliest time covered by IONEX data.
C-----------------------------------------------------------------------
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      IONFIR = MAPTIM(1)
      END
      DOUBLE PRECISION FUNCTION IONLAS()
C-----------------------------------------------------------------------
C   Latest time covered by IONEX data.
C-----------------------------------------------------------------------
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      IONLAS = MAPTIM(TOTMAP)
      END
      REAL FUNCTION IONALT()
C-----------------------------------------------------------------------
C   Altitude of the ionosphere in meters.
C-----------------------------------------------------------------------
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
      IONALT = ALT
      END
      SUBROUTINE IONTEC (LFACT, LAT, LONG, TIME, TEC, IRET)
C-----------------------------------------------------------------------
C   Find the zenith TEC for LAT, LONG at TIME.
C
C   Inputs:
C      LAT     R      Latitude (radians)
C      LONG    R      Longitude (radians)
C      TIME    D      Time (Julian date)
C
C   Outputs:
C      TEC     R      Zenith TEC (electrons per square meter)
C      IRET    I      Return status:
C                      0 - TEC valid
C                      1 - data not available
C-----------------------------------------------------------------------
      REAL      LFACT, LAT, LONG
      DOUBLE PRECISION TIME
      REAL      TEC
      INTEGER   IRET
C
      INCLUDE 'INCS:PSTD.INC'
C
C     Local variables
C
C     TIDX    Index of map at latest epoch less than or equal to TIME
C     DT1     Weight for linear interpolation
C     DT2     Weight for linear interpolation
C     TEC1    Zenith TEC from map before TIME
C     TEC2    Zenith TEC from map after TIME
C     TLONG   Time-shifted longitude (degrees)
C
      INTEGER   TIDX
      DOUBLE PRECISION DT1, DT2
      REAL      TEC1, TEC2, TLONG
C
      INCLUDE 'INCS:DDCH.INC'
      INCLUDE 'IONEX.INC'
C-----------------------------------------------------------------------
C
C     Use linear interpolation between maps bracketing the time at which
C     the zenith TEC is required. This requires maps on either side of
C     the current time.
C
      IF ((MAPTIM(1) .LE. TIME) .AND. (TIME .LT. MAPTIM(TOTMAP))) THEN
         TIDX = 0
C
C        Invariant: MAPTIM(1:TIDX) .LE. TIME
C        Bound:     TOTMAP - TIDX - 1
C
   10    IF (MAPTIM(TIDX + 1) .LE. TIME) THEN
            TIDX = TIDX + 1
            GO TO 10
         END IF
C
C        TIDX < TOTMAP and MAPTIM(TIDX) <= TIME < MAPTIM(TIDX + 1)
C
         DT1 = TIME - MAPTIM(TIDX)
         DT2 = MAPTIM(TIDX+1) - TIME
C
C        The ionosphere rotates with the sun so interpolation should
C        give better results is we adjust the longitude to account for
C        this.
C
C        At the time of the earlier map at TIDX, the ionosphere
C        currently at LONG was DT1 turns to the East (more positive).
C
         TLONG = RAD2DG * LONG + 360.0 * DT1 * LFACT
         IF (TLONG .LT. -180.0) THEN
            TLONG = TLONG + 360.0
         ELSE IF (TLONG .GT. 180.0) THEN
            TLONG = TLONG - 360.0
         END IF
         CALL TECVAL (MAPS(1, 1, TIDX), MAXLON, NLONG, NLAT, MINLON,
     *                DLONG, MINLAT, DLAT, TLONG, REAL(RAD2DG * LAT),
     *                TEC1, IRET)
         IF (IRET .EQ. 0) THEN
C
C           At the the time of the later map at TIDX + 1, the
C           ionosphere at LONG will be DT2 turns to the West (more
C           negative).
C
            TLONG = RAD2DG * LONG - 360.0 * DT2 * LFACT
            IF (TLONG .LT. -180.0) THEN
               TLONG = TLONG + 360.0
            ELSE IF (TLONG .GT. 180.0) THEN
               TLONG = TLONG - 360.0
            END IF
            CALL TECVAL (MAPS(1, 1, TIDX + 1), MAXLON, NLONG, NLAT,
     *                   MINLON, DLONG, MINLAT, DLAT, TLONG,
     *                   REAL(RAD2DG * LAT), TEC2, IRET)
            IF (IRET .EQ. 0) THEN
               TEC = (DT2 * TEC1 + DT1 * TEC2) / (DT1 + DT2)
               IRET = 0
            ELSE
               TEC = FBLANK
               IRET = 1
            END IF
         ELSE
            TEC = FBLANK
            IRET = 1
         END IF
      ELSE
         TEC = FBLANK
         IRET = 1
      END IF
      END
      SUBROUTINE MGRF14 (ISV, DATE, ITYPE, ALT, COLAT, ELONG, X, Y, Z,
     *   F)
C-----------------------------------------------------------------------
C   This subroutine was part of the file igrf14.f and was downloaded
C   from https://www.ngdc.noaa.gov/IAGA/vmod/igrf.html
C   It has been changed to AIPS coding habits.
C   in November 2024 by IAGA Working Group V-MOD.
C   It is the 14th generation IGRF, ie the 13th revision.
C   The main-field models for 1900.0, 1905.0,..1940.0 and 2025.0 are
C   non-definitive, those for 1945.0, 1950.0,...2020.0 are definitive
C   and the secular-variation model for 2025.0 to 2030.0 is
C   non-definitive.
C-----------------------------------------------------------------------
C   This is a synthesis routine for the 14th generation IGRF as agreed
C   in December 2024 by IAGA Working Group V-MOD. It is valid 1900.0 to
C   2030.0 inclusive. Values for dates from 1945.0 to 2020.0 inclusive
C   are definitive, otherwise they are non-definitive.
C   Input
C      isv   = 0 if main-field values are required
C      isv   = 1 if secular variation values are required
C      date  = year A.D. Must be greater than or equal to 1900.0 and
C              less than or equal to 2035.0. Warning message is given
C              for dates greater than 2030.0. Must be double precision.
C      itype = 1 if geodetic (spheroid)
C      itype = 2 if geocentric (sphere)
C      alt   = height in km above sea level if itype = 1
C            = distance from centre of Earth in km if itype = 2 (>3485 km)
C      colat = colatitude (0-180)
C      elong = east-longitude (0-360)
C      alt, colat and elong must be double precision.
C   Output
C      x     = north component (nT) if isv = 0, nT/year if isv = 1
C      y     = east component (nT) if isv = 0, nT/year if isv = 1
C      z     = vertical component (nT) if isv = 0, nT/year if isv = 1
C      f     = total intensity (nT) if isv = 0, rubbish if isv = 1
c
C   To get the other geomagnetic elements (D, I, H and secular
C   variations dD, dH, dI and dF) use routines ptoc and ptocsv.
c
C   Adapted from 8th generation version to include new maximum degree
C   for main-field models for 2000.0 and onwards and use WGS84 spheroid
C   instead of International Astronomical Union 1966 spheroid as
C   recommended by IAGA in July 2003. Reference radius remains as
C   6371.2 km - it is NOT the mean radius (= 6371.0 km) but 6371.2 km is
C   what is used in determining the coefficients.
C   Adaptation by Susan Macmillan, August 2003 (for 9th generation),
C   December 2004, December 2009 & December 2014;
C   by William Brown, December 2019, February 2020. Updated by
C   Ciaran Beggan, November 2024
c
C   Coefficients at 1995.0 incorrectly rounded (rounded up instead of
C   to even) included as these are the coefficients published in Excel
C   spreadsheet July 2005.
C-----------------------------------------------------------------------
C      implicit double precision (a-h,o-z)
      INTEGER   ISV, ITYPE
      DOUBLE PRECISION DATE, ALT, COLAT, ELONG, X, Y, Z, F
C
      INTEGER   I, J, K, KMX, L, LL, LM, M, N, NC, NMX, NMSG
C
      DOUBLE PRECISION GH(3840), G0(120), G1(120), G2(120), G3(120),
     1   G4(120), G5(120), G6(120), G7(120), G8(120), G9(120), GA(120),
     2   GB(120), GC(120), GD(120), GE(120), GF(120), GG(120), GI(120),
     3   GJ(120), GK(195), GL(195), GM(195), GP(195), GQ(195), GR(195),
     4   GS(195), GT(195), P(105), Q(105), CL(13), SL(13)
      EQUIVALENCE (G0,GH(1)), (G1,GH(121)), (G2,GH(241)), (G3,GH(361)),
     1   (G4,GH(481)), (G5,GH(601)), (G6,GH(721)), (G7,GH(841)),
     2   (G8,GH(961)), (G9,GH(1081)), (GA,GH(1201)), (GB,GH(1321)),
     3   (GC,GH(1441)), (GD,GH(1561)), (GE,GH(1681)), (GF,GH(1801)),
     4   (GG,GH(1921)), (GI,GH(2041)), (GJ,GH(2161)), (GK,GH(2281)),
     5   (GL,GH(2476)), (GM,GH(2671)), (GP,GH(2866)), (GQ,GH(3061)),
     6   (GR,GH(3256)), (GS,GH(3451)), (GT,GH(3646))
      DOUBLE PRECISION A2, B2, CD, CT, FM, FN, GMM, GN, ONE, R, RATIO,
     *   RHO, RR, SD, ST, T, TC, THREE, TWO
      SAVE NMSG
      DATA NMSG /0/
C
      INCLUDE 'INCS:DMSG.INC'
C
      DATA G0/ -31543.,-2298., 5922., -677., 2905.,-1061.,  924., 1121., 1900
     1           1022.,-1469., -330., 1256.,    3.,  572.,  523.,  876., 1900
     2            628.,  195.,  660.,  -69., -361., -210.,  134.,  -75., 1900
     3           -184.,  328., -210.,  264.,   53.,    5.,  -33.,  -86., 1900
     4           -124.,  -16.,    3.,   63.,   61.,   -9.,  -11.,   83., 1900
     5           -217.,    2.,  -58.,  -35.,   59.,   36.,  -90.,  -69., 1900
     6             70.,  -55.,  -45.,    0.,  -13.,   34.,  -10.,  -41., 1900
     7             -1.,  -21.,   28.,   18.,  -12.,    6.,  -22.,   11., 1900
     8              8.,    8.,   -4.,  -14.,   -9.,    7.,    1.,  -13., 1900
     9              2.,    5.,   -9.,   16.,    5.,   -5.,    8.,  -18., 1900
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1900
     b             -3.,    1.,   -2.,   -2.,    8.,    2.,   10.,   -1., 1900
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1900
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1900
     e              0.,   -2.,    2.,    4.,    2.,    0.,    0.,   -6./ 1900
      DATA G1/ -31464.,-2298., 5909., -728., 2928.,-1086., 1041., 1065., 1905
     1           1037.,-1494., -357., 1239.,   34.,  635.,  480.,  880., 1905
     2            643.,  203.,  653.,  -77., -380., -201.,  146.,  -65., 1905
     3           -192.,  328., -193.,  259.,   56.,   -1.,  -32.,  -93., 1905
     4           -125.,  -26.,   11.,   62.,   60.,   -7.,  -11.,   86., 1905
     5           -221.,    4.,  -57.,  -32.,   57.,   32.,  -92.,  -67., 1905
     6             70.,  -54.,  -46.,    0.,  -14.,   33.,  -11.,  -41., 1905
     7              0.,  -20.,   28.,   18.,  -12.,    6.,  -22.,   11., 1905
     8              8.,    8.,   -4.,  -15.,   -9.,    7.,    1.,  -13., 1905
     9              2.,    5.,   -8.,   16.,    5.,   -5.,    8.,  -18., 1905
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1905
     b             -3.,    1.,   -2.,   -2.,    8.,    2.,   10.,    0., 1905
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1905
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1905
     e              0.,   -2.,    2.,    4.,    2.,    0.,    0.,   -6./ 1905
      DATA G2/ -31354.,-2297., 5898., -769., 2948.,-1128., 1176., 1000., 1910
     1           1058.,-1524., -389., 1223.,   62.,  705.,  425.,  884., 1910
     2            660.,  211.,  644.,  -90., -400., -189.,  160.,  -55., 1910
     3           -201.,  327., -172.,  253.,   57.,   -9.,  -33., -102., 1910
     4           -126.,  -38.,   21.,   62.,   58.,   -5.,  -11.,   89., 1910
     5           -224.,    5.,  -54.,  -29.,   54.,   28.,  -95.,  -65., 1910
     6             71.,  -54.,  -47.,    1.,  -14.,   32.,  -12.,  -40., 1910
     7              1.,  -19.,   28.,   18.,  -13.,    6.,  -22.,   11., 1910
     8              8.,    8.,   -4.,  -15.,   -9.,    6.,    1.,  -13., 1910
     9              2.,    5.,   -8.,   16.,    5.,   -5.,    8.,  -18., 1910
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1910
     b             -3.,    1.,   -2.,   -2.,    8.,    2.,   10.,    0., 1910
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1910
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1910
     e              0.,   -2.,    2.,    4.,    2.,    0.,    0.,   -6./ 1910
      DATA G3/ -31212.,-2306., 5875., -802., 2956.,-1191., 1309.,  917., 1915
     1           1084.,-1559., -421., 1212.,   84.,  778.,  360.,  887., 1915
     2            678.,  218.,  631., -109., -416., -173.,  178.,  -51., 1915
     3           -211.,  327., -148.,  245.,   58.,  -16.,  -34., -111., 1915
     4           -126.,  -51.,   32.,   61.,   57.,   -2.,  -10.,   93., 1915
     5           -228.,    8.,  -51.,  -26.,   49.,   23.,  -98.,  -62., 1915
     6             72.,  -54.,  -48.,    2.,  -14.,   31.,  -12.,  -38., 1915
     7              2.,  -18.,   28.,   19.,  -15.,    6.,  -22.,   11., 1915
     8              8.,    8.,   -4.,  -15.,   -9.,    6.,    2.,  -13., 1915
     9              3.,    5.,   -8.,   16.,    6.,   -5.,    8.,  -18., 1915
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1915
     b             -3.,    1.,   -2.,   -2.,    8.,    2.,   10.,    0., 1915
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1915
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1915
     e              0.,   -2.,    1.,    4.,    2.,    0.,    0.,   -6./ 1915
      DATA G4/ -31060.,-2317., 5845., -839., 2959.,-1259., 1407.,  823., 1920
     1           1111.,-1600., -445., 1205.,  103.,  839.,  293.,  889., 1920
     2            695.,  220.,  616., -134., -424., -153.,  199.,  -57., 1920
     3           -221.,  326., -122.,  236.,   58.,  -23.,  -38., -119., 1920
     4           -125.,  -62.,   43.,   61.,   55.,    0.,  -10.,   96., 1920
     5           -233.,   11.,  -46.,  -22.,   44.,   18., -101.,  -57., 1920
     6             73.,  -54.,  -49.,    2.,  -14.,   29.,  -13.,  -37., 1920
     7              4.,  -16.,   28.,   19.,  -16.,    6.,  -22.,   11., 1920
     8              7.,    8.,   -3.,  -15.,   -9.,    6.,    2.,  -14., 1920
     9              4.,    5.,   -7.,   17.,    6.,   -5.,    8.,  -19., 1920
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1920
     b             -3.,    1.,   -2.,   -2.,    9.,    2.,   10.,    0., 1920
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1920
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1920
     e              0.,   -2.,    1.,    4.,    3.,    0.,    0.,   -6./ 1920
      DATA G5/ -30926.,-2318., 5817., -893., 2969.,-1334., 1471.,  728., 1925
     1           1140.,-1645., -462., 1202.,  119.,  881.,  229.,  891., 1925
     2            711.,  216.,  601., -163., -426., -130.,  217.,  -70., 1925
     3           -230.,  326.,  -96.,  226.,   58.,  -28.,  -44., -125., 1925
     4           -122.,  -69.,   51.,   61.,   54.,    3.,   -9.,   99., 1925
     5           -238.,   14.,  -40.,  -18.,   39.,   13., -103.,  -52., 1925
     6             73.,  -54.,  -50.,    3.,  -14.,   27.,  -14.,  -35., 1925
     7              5.,  -14.,   29.,   19.,  -17.,    6.,  -21.,   11., 1925
     8              7.,    8.,   -3.,  -15.,   -9.,    6.,    2.,  -14., 1925
     9              4.,    5.,   -7.,   17.,    7.,   -5.,    8.,  -19., 1925
     a              8.,   10.,  -20.,    1.,   14.,  -11.,    5.,   12., 1925
     b             -3.,    1.,   -2.,   -2.,    9.,    2.,   10.,    0., 1925
     c             -2.,   -1.,    2.,   -3.,   -4.,    2.,    2.,    1., 1925
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1925
     e              0.,   -2.,    1.,    4.,    3.,    0.,    0.,   -6./ 1925
      DATA G6/ -30805.,-2316., 5808., -951., 2980.,-1424., 1517.,  644., 1930
     1           1172.,-1692., -480., 1205.,  133.,  907.,  166.,  896., 1930
     2            727.,  205.,  584., -195., -422., -109.,  234.,  -90., 1930
     3           -237.,  327.,  -72.,  218.,   60.,  -32.,  -53., -131., 1930
     4           -118.,  -74.,   58.,   60.,   53.,    4.,   -9.,  102., 1930
     5           -242.,   19.,  -32.,  -16.,   32.,    8., -104.,  -46., 1930
     6             74.,  -54.,  -51.,    4.,  -15.,   25.,  -14.,  -34., 1930
     7              6.,  -12.,   29.,   18.,  -18.,    6.,  -20.,   11., 1930
     8              7.,    8.,   -3.,  -15.,   -9.,    5.,    2.,  -14., 1930
     9              5.,    5.,   -6.,   18.,    8.,   -5.,    8.,  -19., 1930
     a              8.,   10.,  -20.,    1.,   14.,  -12.,    5.,   12., 1930
     b             -3.,    1.,   -2.,   -2.,    9.,    3.,   10.,    0., 1930
     c             -2.,   -2.,    2.,   -3.,   -4.,    2.,    2.,    1., 1930
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1930
     e              0.,   -2.,    1.,    4.,    3.,    0.,    0.,   -6./ 1930
      DATA G7/ -30715.,-2306., 5812.,-1018., 2984.,-1520., 1550.,  586., 1935
     1           1206.,-1740., -494., 1215.,  146.,  918.,  101.,  903., 1935
     2            744.,  188.,  565., -226., -415.,  -90.,  249., -114., 1935
     3           -241.,  329.,  -51.,  211.,   64.,  -33.,  -64., -136., 1935
     4           -115.,  -76.,   64.,   59.,   53.,    4.,   -8.,  104., 1935
     5           -246.,   25.,  -25.,  -15.,   25.,    4., -106.,  -40., 1935
     6             74.,  -53.,  -52.,    4.,  -17.,   23.,  -14.,  -33., 1935
     7              7.,  -11.,   29.,   18.,  -19.,    6.,  -19.,   11., 1935
     8              7.,    8.,   -3.,  -15.,   -9.,    5.,    1.,  -15., 1935
     9              6.,    5.,   -6.,   18.,    8.,   -5.,    7.,  -19., 1935
     a              8.,   10.,  -20.,    1.,   15.,  -12.,    5.,   11., 1935
     b             -3.,    1.,   -3.,   -2.,    9.,    3.,   11.,    0., 1935
     c             -2.,   -2.,    2.,   -3.,   -4.,    2.,    2.,    1., 1935
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1935
     e              0.,   -1.,    2.,    4.,    3.,    0.,    0.,   -6./ 1935
      DATA G8/ -30654.,-2292., 5821.,-1106., 2981.,-1614., 1566.,  528., 1940
     1           1240.,-1790., -499., 1232.,  163.,  916.,   43.,  914., 1940
     2            762.,  169.,  550., -252., -405.,  -72.,  265., -141., 1940
     3           -241.,  334.,  -33.,  208.,   71.,  -33.,  -75., -141., 1940
     4           -113.,  -76.,   69.,   57.,   54.,    4.,   -7.,  105., 1940
     5           -249.,   33.,  -18.,  -15.,   18.,    0., -107.,  -33., 1940
     6             74.,  -53.,  -52.,    4.,  -18.,   20.,  -14.,  -31., 1940
     7              7.,   -9.,   29.,   17.,  -20.,    5.,  -19.,   11., 1940
     8              7.,    8.,   -3.,  -14.,  -10.,    5.,    1.,  -15., 1940
     9              6.,    5.,   -5.,   19.,    9.,   -5.,    7.,  -19., 1940
     a              8.,   10.,  -21.,    1.,   15.,  -12.,    5.,   11., 1940
     b             -3.,    1.,   -3.,   -2.,    9.,    3.,   11.,    1., 1940
     c             -2.,   -2.,    2.,   -3.,   -4.,    2.,    2.,    1., 1940
     d             -5.,    2.,   -2.,    6.,    6.,   -4.,    4.,    0., 1940
     e              0.,   -1.,    2.,    4.,    3.,    0.,    0.,   -6./ 1940
      DATA G9/ -30594.,-2285., 5810.,-1244., 2990.,-1702., 1578.,  477., 1945
     1           1282.,-1834., -499., 1255.,  186.,  913.,  -11.,  944., 1945
     2            776.,  144.,  544., -276., -421.,  -55.,  304., -178., 1945
     3           -253.,  346.,  -12.,  194.,   95.,  -20.,  -67., -142., 1945
     4           -119.,  -82.,   82.,   59.,   57.,    6.,    6.,  100., 1945
     5           -246.,   16.,  -25.,   -9.,   21.,  -16., -104.,  -39., 1945
     6             70.,  -40.,  -45.,    0.,  -18.,    0.,    2.,  -29., 1945
     7              6.,  -10.,   28.,   15.,  -17.,   29.,  -22.,   13., 1945
     8              7.,   12.,   -8.,  -21.,   -5.,  -12.,    9.,   -7., 1945
     9              7.,    2.,  -10.,   18.,    7.,    3.,    2.,  -11., 1945
     a              5.,  -21.,  -27.,    1.,   17.,  -11.,   29.,    3., 1945
     b             -9.,   16.,    4.,   -3.,    9.,   -4.,    6.,   -3., 1945
     c              1.,   -4.,    8.,   -3.,   11.,    5.,    1.,    1., 1945
     d              2.,  -20.,   -5.,   -1.,   -1.,   -6.,    8.,    6., 1945
     e             -1.,   -4.,   -3.,   -2.,    5.,    0.,   -2.,   -2./ 1945
      DATA GA/ -30554.,-2250., 5815.,-1341., 2998.,-1810., 1576.,  381., 1950
     1           1297.,-1889., -476., 1274.,  206.,  896.,  -46.,  954., 1950
     2            792.,  136.,  528., -278., -408.,  -37.,  303., -210., 1950
     3           -240.,  349.,    3.,  211.,  103.,  -20.,  -87., -147., 1950
     4           -122.,  -76.,   80.,   54.,   57.,   -1.,    4.,   99., 1950
     5           -247.,   33.,  -16.,  -12.,   12.,  -12., -105.,  -30., 1950
     6             65.,  -55.,  -35.,    2.,  -17.,    1.,    0.,  -40., 1950
     7             10.,   -7.,   36.,    5.,  -18.,   19.,  -16.,   22., 1950
     8             15.,    5.,   -4.,  -22.,   -1.,    0.,   11.,  -21., 1950
     9             15.,   -8.,  -13.,   17.,    5.,   -4.,   -1.,  -17., 1950
     a              3.,   -7.,  -24.,   -1.,   19.,  -25.,   12.,   10., 1950
     b              2.,    5.,    2.,   -5.,    8.,   -2.,    8.,    3., 1950
     c            -11.,    8.,   -7.,   -8.,    4.,   13.,   -1.,   -2., 1950
     d             13.,  -10.,   -4.,    2.,    4.,   -3.,   12.,    6., 1950
     e              3.,   -3.,    2.,    6.,   10.,   11.,    3.,    8./ 1950
      DATA GB/ -30500.,-2215., 5820.,-1440., 3003.,-1898., 1581.,  291., 1955
     1           1302.,-1944., -462., 1288.,  216.,  882.,  -83.,  958., 1955
     2            796.,  133.,  510., -274., -397.,  -23.,  290., -230., 1955
     3           -229.,  360.,   15.,  230.,  110.,  -23.,  -98., -152., 1955
     4           -121.,  -69.,   78.,   47.,   57.,   -9.,    3.,   96., 1955
     5           -247.,   48.,   -8.,  -16.,    7.,  -12., -107.,  -24., 1955
     6             65.,  -56.,  -50.,    2.,  -24.,   10.,   -4.,  -32., 1955
     7              8.,  -11.,   28.,    9.,  -20.,   18.,  -18.,   11., 1955
     8              9.,   10.,   -6.,  -15.,  -14.,    5.,    6.,  -23., 1955
     9             10.,    3.,   -7.,   23.,    6.,   -4.,    9.,  -13., 1955
     a              4.,    9.,  -11.,   -4.,   12.,   -5.,    7.,    2., 1955
     b              6.,    4.,   -2.,    1.,   10.,    2.,    7.,    2., 1955
     c             -6.,    5.,    5.,   -3.,   -5.,   -4.,   -1.,    0., 1955
     d              2.,   -8.,   -3.,   -2.,    7.,   -4.,    4.,    1., 1955
     e             -2.,   -3.,    6.,    7.,   -2.,   -1.,    0.,   -3./ 1955
      DATA GC/ -30421.,-2169., 5791.,-1555., 3002.,-1967., 1590.,  206., 1960
     1           1302.,-1992., -414., 1289.,  224.,  878., -130.,  957., 1960
     2            800.,  135.,  504., -278., -394.,    3.,  269., -255., 1960
     3           -222.,  362.,   16.,  242.,  125.,  -26., -117., -156., 1960
     4           -114.,  -63.,   81.,   46.,   58.,  -10.,    1.,   99., 1960
     5           -237.,   60.,   -1.,  -20.,   -2.,  -11., -113.,  -17., 1960
     6             67.,  -56.,  -55.,    5.,  -28.,   15.,   -6.,  -32., 1960
     7              7.,   -7.,   23.,   17.,  -18.,    8.,  -17.,   15., 1960
     8              6.,   11.,   -4.,  -14.,  -11.,    7.,    2.,  -18., 1960
     9             10.,    4.,   -5.,   23.,   10.,    1.,    8.,  -20., 1960
     a              4.,    6.,  -18.,    0.,   12.,   -9.,    2.,    1., 1960
     b              0.,    4.,   -3.,   -1.,    9.,   -2.,    8.,    3., 1960
     c              0.,   -1.,    5.,    1.,   -3.,    4.,    4.,    1., 1960
     d              0.,    0.,   -1.,    2.,    4.,   -5.,    6.,    1., 1960
     e              1.,   -1.,   -1.,    6.,    2.,    0.,    0.,   -7./ 1960
      DATA GD/ -30334.,-2119., 5776.,-1662., 2997.,-2016., 1594.,  114., 1965
     1           1297.,-2038., -404., 1292.,  240.,  856., -165.,  957., 1965
     2            804.,  148.,  479., -269., -390.,   13.,  252., -269., 1965
     3           -219.,  358.,   19.,  254.,  128.,  -31., -126., -157., 1965
     4            -97.,  -62.,   81.,   45.,   61.,  -11.,    8.,  100., 1965
     5           -228.,   68.,    4.,  -32.,    1.,   -8., -111.,   -7., 1965
     6             75.,  -57.,  -61.,    4.,  -27.,   13.,   -2.,  -26., 1965
     7              6.,   -6.,   26.,   13.,  -23.,    1.,  -12.,   13., 1965
     8              5.,    7.,   -4.,  -12.,  -14.,    9.,    0.,  -16., 1965
     9              8.,    4.,   -1.,   24.,   11.,   -3.,    4.,  -17., 1965
     a              8.,   10.,  -22.,    2.,   15.,  -13.,    7.,   10., 1965
     b             -4.,   -1.,   -5.,   -1.,   10.,    5.,   10.,    1., 1965
     c             -4.,   -2.,    1.,   -2.,   -3.,    2.,    2.,    1., 1965
     d             -5.,    2.,   -2.,    6.,    4.,   -4.,    4.,    0., 1965
     e              0.,   -2.,    2.,    3.,    2.,    0.,    0.,   -6./ 1965
      DATA GE/ -30220.,-2068., 5737.,-1781., 3000.,-2047., 1611.,   25., 1970
     1           1287.,-2091., -366., 1278.,  251.,  838., -196.,  952., 1970
     2            800.,  167.,  461., -266., -395.,   26.,  234., -279., 1970
     3           -216.,  359.,   26.,  262.,  139.,  -42., -139., -160., 1970
     4            -91.,  -56.,   83.,   43.,   64.,  -12.,   15.,  100., 1970
     5           -212.,   72.,    2.,  -37.,    3.,   -6., -112.,    1., 1970
     6             72.,  -57.,  -70.,    1.,  -27.,   14.,   -4.,  -22., 1970
     7              8.,   -2.,   23.,   13.,  -23.,   -2.,  -11.,   14., 1970
     8              6.,    7.,   -2.,  -15.,  -13.,    6.,   -3.,  -17., 1970
     9              5.,    6.,    0.,   21.,   11.,   -6.,    3.,  -16., 1970
     a              8.,   10.,  -21.,    2.,   16.,  -12.,    6.,   10., 1970
     b             -4.,   -1.,   -5.,    0.,   10.,    3.,   11.,    1., 1970
     c             -2.,   -1.,    1.,   -3.,   -3.,    1.,    2.,    1., 1970
     d             -5.,    3.,   -1.,    4.,    6.,   -4.,    4.,    0., 1970
     e              1.,   -1.,    0.,    3.,    3.,    1.,   -1.,   -4./ 1970
      DATA GF/ -30100.,-2013., 5675.,-1902., 3010.,-2067., 1632.,  -68., 1975
     1           1276.,-2144., -333., 1260.,  262.,  830., -223.,  946., 1975
     2            791.,  191.,  438., -265., -405.,   39.,  216., -288., 1975
     3           -218.,  356.,   31.,  264.,  148.,  -59., -152., -159., 1975
     4            -83.,  -49.,   88.,   45.,   66.,  -13.,   28.,   99., 1975
     5           -198.,   75.,    1.,  -41.,    6.,   -4., -111.,   11., 1975
     6             71.,  -56.,  -77.,    1.,  -26.,   16.,   -5.,  -14., 1975
     7             10.,    0.,   22.,   12.,  -23.,   -5.,  -12.,   14., 1975
     8              6.,    6.,   -1.,  -16.,  -12.,    4.,   -8.,  -19., 1975
     9              4.,    6.,    0.,   18.,   10.,  -10.,    1.,  -17., 1975
     a              7.,   10.,  -21.,    2.,   16.,  -12.,    7.,   10., 1975
     b             -4.,   -1.,   -5.,   -1.,   10.,    4.,   11.,    1., 1975
     c             -3.,   -2.,    1.,   -3.,   -3.,    1.,    2.,    1., 1975
     d             -5.,    3.,   -2.,    4.,    5.,   -4.,    4.,   -1., 1975
     e              1.,   -1.,    0.,    3.,    3.,    1.,   -1.,   -5./ 1975
      DATA GG/ -29992.,-1956., 5604.,-1997., 3027.,-2129., 1663., -200., 1980
     1           1281.,-2180., -336., 1251.,  271.,  833., -252.,  938., 1980
     2            782.,  212.,  398., -257., -419.,   53.,  199., -297., 1980
     3           -218.,  357.,   46.,  261.,  150.,  -74., -151., -162., 1980
     4            -78.,  -48.,   92.,   48.,   66.,  -15.,   42.,   93., 1980
     5           -192.,   71.,    4.,  -43.,   14.,   -2., -108.,   17., 1980
     6             72.,  -59.,  -82.,    2.,  -27.,   21.,   -5.,  -12., 1980
     7             16.,    1.,   18.,   11.,  -23.,   -2.,  -10.,   18., 1980
     8              6.,    7.,    0.,  -18.,  -11.,    4.,   -7.,  -22., 1980
     9              4.,    9.,    3.,   16.,    6.,  -13.,   -1.,  -15., 1980
     a              5.,   10.,  -21.,    1.,   16.,  -12.,    9.,    9., 1980
     b             -5.,   -3.,   -6.,   -1.,    9.,    7.,   10.,    2., 1980
     c             -6.,   -5.,    2.,   -4.,   -4.,    1.,    2.,    0., 1980
     d             -5.,    3.,   -2.,    6.,    5.,   -4.,    3.,    0., 1980
     e              1.,   -1.,    2.,    4.,    3.,    0.,    0.,   -6./ 1980
      DATA GI/ -29873.,-1905., 5500.,-2072., 3044.,-2197., 1687., -306., 1985
     1           1296.,-2208., -310., 1247.,  284.,  829., -297.,  936., 1985
     2            780.,  232.,  361., -249., -424.,   69.,  170., -297., 1985
     3           -214.,  355.,   47.,  253.,  150.,  -93., -154., -164., 1985
     4            -75.,  -46.,   95.,   53.,   65.,  -16.,   51.,   88., 1985
     5           -185.,   69.,    4.,  -48.,   16.,   -1., -102.,   21., 1985
     6             74.,  -62.,  -83.,    3.,  -27.,   24.,   -2.,   -6., 1985
     7             20.,    4.,   17.,   10.,  -23.,    0.,   -7.,   21., 1985
     8              6.,    8.,    0.,  -19.,  -11.,    5.,   -9.,  -23., 1985
     9              4.,   11.,    4.,   14.,    4.,  -15.,   -4.,  -11., 1985
     a              5.,   10.,  -21.,    1.,   15.,  -12.,    9.,    9., 1985
     b             -6.,   -3.,   -6.,   -1.,    9.,    7.,    9.,    1., 1985
     c             -7.,   -5.,    2.,   -4.,   -4.,    1.,    3.,    0., 1985
     d             -5.,    3.,   -2.,    6.,    5.,   -4.,    3.,    0., 1985
     e              1.,   -1.,    2.,    4.,    3.,    0.,    0.,   -6./ 1985
      DATA GJ/ -29775.,-1848., 5406.,-2131., 3059.,-2279., 1686., -373., 1990
     1           1314.,-2239., -284., 1248.,  293.,  802., -352.,  939., 1990
     2            780.,  247.,  325., -240., -423.,   84.,  141., -299., 1990
     3           -214.,  353.,   46.,  245.,  154., -109., -153., -165., 1990
     4            -69.,  -36.,   97.,   61.,   65.,  -16.,   59.,   82., 1990
     5           -178.,   69.,    3.,  -52.,   18.,    1.,  -96.,   24., 1990
     6             77.,  -64.,  -80.,    2.,  -26.,   26.,    0.,   -1., 1990
     7             21.,    5.,   17.,    9.,  -23.,    0.,   -4.,   23., 1990
     8              5.,   10.,   -1.,  -19.,  -10.,    6.,  -12.,  -22., 1990
     9              3.,   12.,    4.,   12.,    2.,  -16.,   -6.,  -10., 1990
     a              4.,    9.,  -20.,    1.,   15.,  -12.,   11.,    9., 1990
     b             -7.,   -4.,   -7.,   -2.,    9.,    7.,    8.,    1., 1990
     c             -7.,   -6.,    2.,   -3.,   -4.,    2.,    2.,    1., 1990
     d             -5.,    3.,   -2.,    6.,    4.,   -4.,    3.,    0., 1990
     e              1.,   -2.,    3.,    3.,    3.,   -1.,    0.,   -6./ 1990
      DATA GK/ -29692.,-1784., 5306.,-2200., 3070.,-2366., 1681., -413., 1995
     1           1335.,-2267., -262., 1249.,  302.,  759., -427.,  940., 1995
     2            780.,  262.,  290., -236., -418.,   97.,  122., -306., 1995
     3           -214.,  352.,   46.,  235.,  165., -118., -143., -166., 1995
     4            -55.,  -17.,  107.,   68.,   67.,  -17.,   68.,   72., 1995
     5           -170.,   67.,   -1.,  -58.,   19.,    1.,  -93.,   36., 1995
     6             77.,  -72.,  -69.,    1.,  -25.,   28.,    4.,    5., 1995
     7             24.,    4.,   17.,    8.,  -24.,   -2.,   -6.,   25., 1995
     8              6.,   11.,   -6.,  -21.,   -9.,    8.,  -14.,  -23., 1995
     9              9.,   15.,    6.,   11.,   -5.,  -16.,   -7.,   -4., 1995
     a              4.,    9.,  -20.,    3.,   15.,  -10.,   12.,    8., 1995
     b             -6.,   -8.,   -8.,   -1.,    8.,   10.,    5.,   -2., 1995
     c             -8.,   -8.,    3.,   -3.,   -6.,    1.,    2.,    0., 1995
     d             -4.,    4.,   -1.,    5.,    4.,   -5.,    2.,   -1., 1995
     e              2.,   -2.,    5.,    1.,    1.,   -2.,    0.,   -7., 1995
     f           75*0./                                                  1995
      DATA GL/ -29619.4,-1728.2, 5186.1,-2267.7, 3068.4,-2481.6, 1670.9, 2000
     1           -458.0, 1339.6,-2288.0, -227.6, 1252.1,  293.4,  714.5, 2000
     2           -491.1,  932.3,  786.8,  272.6,  250.0, -231.9, -403.0, 2000
     3            119.8,  111.3, -303.8, -218.8,  351.4,   43.8,  222.3, 2000
     4            171.9, -130.4, -133.1, -168.6,  -39.3,  -12.9,  106.3, 2000
     5             72.3,   68.2,  -17.4,   74.2,   63.7, -160.9,   65.1, 2000
     6             -5.9,  -61.2,   16.9,    0.7,  -90.4,   43.8,   79.0, 2000
     7            -74.0,  -64.6,    0.0,  -24.2,   33.3,    6.2,    9.1, 2000
     8             24.0,    6.9,   14.8,    7.3,  -25.4,   -1.2,   -5.8, 2000
     9             24.4,    6.6,   11.9,   -9.2,  -21.5,   -7.9,    8.5, 2000
     a            -16.6,  -21.5,    9.1,   15.5,    7.0,    8.9,   -7.9, 2000
     b            -14.9,   -7.0,   -2.1,    5.0,    9.4,  -19.7,    3.0, 2000
     c             13.4,   -8.4,   12.5,    6.3,   -6.2,   -8.9,   -8.4, 2000
     d             -1.5,    8.4,    9.3,    3.8,   -4.3,   -8.2,   -8.2, 2000
     e              4.8,   -2.6,   -6.0,    1.7,    1.7,    0.0,   -3.1, 2000
     f              4.0,   -0.5,    4.9,    3.7,   -5.9,    1.0,   -1.2, 2000
     g              2.0,   -2.9,    4.2,    0.2,    0.3,   -2.2,   -1.1, 2000
     h             -7.4,    2.7,   -1.7,    0.1,   -1.9,    1.3,    1.5, 2000
     i             -0.9,   -0.1,   -2.6,    0.1,    0.9,   -0.7,   -0.7, 2000
     j              0.7,   -2.8,    1.7,   -0.9,    0.1,   -1.2,    1.2, 2000
     k             -1.9,    4.0,   -0.9,   -2.2,   -0.3,   -0.4,    0.2, 2000
     l              0.3,    0.9,    2.5,   -0.2,   -2.6,    0.9,    0.7, 2000
     m             -0.5,    0.3,    0.3,    0.0,   -0.3,    0.0,   -0.4, 2000
     n              0.3,   -0.1,   -0.9,   -0.2,   -0.4,   -0.4,    0.8, 2000
     o             -0.2,   -0.9,   -0.9,    0.3,    0.2,    0.1,    1.8, 2000
     p             -0.4,   -0.4,    1.3,   -1.0,   -0.4,   -0.1,    0.7, 2000
     q              0.7,   -0.4,    0.3,    0.3,    0.6,   -0.1,    0.3, 2000
     r              0.4,   -0.2,    0.0,   -0.5,    0.1,   -0.9/         2000
      DATA GM/-29554.63,-1669.05, 5077.99,-2337.24, 3047.69,-2594.50,    2005
     1          1657.76, -515.43, 1336.30,-2305.83, -198.86, 1246.39,    2005
     2           269.72,  672.51, -524.72,  920.55,  797.96,  282.07,    2005
     3           210.65, -225.23, -379.86,  145.15,  100.00, -305.36,    2005
     4          -227.00,  354.41,   42.72,  208.95,  180.25, -136.54,    2005
     5          -123.45, -168.05,  -19.57,  -13.55,  103.85,   73.60,    2005
     6            69.56,  -20.33,   76.74,   54.75, -151.34,   63.63,    2005
     7           -14.58,  -63.53,   14.58,    0.24,  -86.36,   50.94,    2005
     8            79.88,  -74.46,  -61.14,   -1.65,  -22.57,   38.73,    2005
     9             6.82,   12.30,   25.35,    9.37,   10.93,    5.42,    2005
     a           -26.32,    1.94,   -4.64,   24.80,    7.62,   11.20,    2005
     b           -11.73,  -20.88,   -6.88,    9.83,  -18.11,  -19.71,    2005
     c            10.17,   16.22,    9.36,    7.61,  -11.25,  -12.76,    2005
     d            -4.87,   -0.06,    5.58,    9.76,  -20.11,    3.58,    2005
     e            12.69,   -6.94,   12.67,    5.01,   -6.72,  -10.76,    2005
     f            -8.16,   -1.25,    8.10,    8.76,    2.92,   -6.66,    2005
     g            -7.73,   -9.22,    6.01,   -2.17,   -6.12,    2.19,    2005
     h             1.42,    0.10,   -2.35,    4.46,   -0.15,    4.76,    2005
     i             3.06,   -6.58,    0.29,   -1.01,    2.06,   -3.47,    2005
     j             3.77,   -0.86,   -0.21,   -2.31,   -2.09,   -7.93,    2005
     k             2.95,   -1.60,    0.26,   -1.88,    1.44,    1.44,    2005
     l            -0.77,   -0.31,   -2.27,    0.29,    0.90,   -0.79,    2005
     m            -0.58,    0.53,   -2.69,    1.80,   -1.08,    0.16,    2005
     n            -1.58,    0.96,   -1.90,    3.99,   -1.39,   -2.15,    2005
     o            -0.29,   -0.55,    0.21,    0.23,    0.89,    2.38,    2005
     p            -0.38,   -2.63,    0.96,    0.61,   -0.30,    0.40,    2005
     q             0.46,    0.01,   -0.35,    0.02,   -0.36,    0.28,    2005
     r             0.08,   -0.87,   -0.49,   -0.34,   -0.08,    0.88,    2005
     s            -0.16,   -0.88,   -0.76,    0.30,    0.33,    0.28,    2005
     t             1.72,   -0.43,   -0.54,    1.18,   -1.07,   -0.37,    2005
     u            -0.04,    0.75,    0.63,   -0.26,    0.21,    0.35,    2005
     v             0.53,   -0.05,    0.38,    0.41,   -0.22,   -0.10,    2005
     w            -0.57,   -0.18,   -0.82/                               2005
      DATA GP/-29496.57,-1586.42, 4944.26,-2396.06, 3026.34,-2708.54,    2010
     1          1668.17, -575.73, 1339.85,-2326.54, -160.40, 1232.10,    2010
     2           251.75,  633.73, -537.03,  912.66,  808.97,  286.48,    2010
     3           166.58, -211.03, -356.83,  164.46,   89.40, -309.72,    2010
     4          -230.87,  357.29,   44.58,  200.26,  189.01, -141.05,    2010
     5          -118.06, -163.17,   -0.01,   -8.03,  101.04,   72.78,    2010
     6            68.69,  -20.90,   75.92,   44.18, -141.40,   61.54,    2010
     7           -22.83,  -66.26,   13.10,    3.02,  -78.09,   55.40,    2010
     8            80.44,  -75.00,  -57.80,   -4.55,  -21.20,   45.24,    2010
     9             6.54,   14.00,   24.96,   10.46,    7.03,    1.64,    2010
     a           -27.61,    4.92,   -3.28,   24.41,    8.21,   10.84,    2010
     b           -14.50,  -20.03,   -5.59,   11.83,  -19.34,  -17.41,    2010
     c            11.61,   16.71,   10.85,    6.96,  -14.05,  -10.74,    2010
     d            -3.54,    1.64,    5.50,    9.45,  -20.54,    3.45,    2010
     e            11.51,   -5.27,   12.75,    3.13,   -7.14,  -12.38,    2010
     f            -7.42,   -0.76,    7.97,    8.43,    2.14,   -8.42,    2010
     g            -6.08,  -10.08,    7.01,   -1.94,   -6.24,    2.73,    2010
     h             0.89,   -0.10,   -1.07,    4.71,   -0.16,    4.44,    2010
     i             2.45,   -7.22,   -0.33,   -0.96,    2.13,   -3.95,    2010
     j             3.09,   -1.99,   -1.03,   -1.97,   -2.80,   -8.31,    2010
     k             3.05,   -1.48,    0.13,   -2.03,    1.67,    1.65,    2010
     l            -0.66,   -0.51,   -1.76,    0.54,    0.85,   -0.79,    2010
     m            -0.39,    0.37,   -2.51,    1.79,   -1.27,    0.12,    2010
     n            -2.11,    0.75,   -1.94,    3.75,   -1.86,   -2.12,    2010
     o            -0.21,   -0.87,    0.30,    0.27,    1.04,    2.13,    2010
     p            -0.63,   -2.49,    0.95,    0.49,   -0.11,    0.59,    2010
     q             0.52,    0.00,   -0.39,    0.13,   -0.37,    0.27,    2010
     r             0.21,   -0.86,   -0.77,   -0.23,    0.04,    0.87,    2010
     s            -0.09,   -0.89,   -0.87,    0.31,    0.30,    0.42,    2010
     t             1.66,   -0.45,   -0.59,    1.08,   -1.14,   -0.31,    2010
     u            -0.07,    0.78,    0.54,   -0.18,    0.10,    0.38,    2010
     v             0.49,    0.02,    0.44,    0.42,   -0.25,   -0.26,    2010
     w            -0.53,   -0.26,   -0.79/                               2010
      DATA GQ/-29441.46,-1501.77, 4795.99,-2445.88, 3012.20,-2845.41,    2015
     1          1676.35, -642.17, 1350.33,-2352.26, -115.29, 1225.85,    2015
     2           245.04,  581.69, -538.70,  907.42,  813.68,  283.54,    2015
     3           120.49, -188.43, -334.85,  180.95,   70.38, -329.23,    2015
     4          -232.91,  360.14,   46.98,  192.35,  196.98, -140.94,    2015
     5          -119.14, -157.40,   15.98,    4.30,  100.12,   69.55,    2015
     6            67.57,  -20.61,   72.79,   33.30, -129.85,   58.74,    2015
     7           -28.93,  -66.64,   13.14,    7.35,  -70.85,   62.41,    2015
     8            81.29,  -75.99,  -54.27,   -6.79,  -19.53,   51.82,    2015
     9             5.59,   15.07,   24.45,    9.32,    3.27,   -2.88,    2015
     a           -27.50,    6.61,   -2.32,   23.98,    8.89,   10.04,    2015
     b           -16.78,  -18.26,   -3.16,   13.18,  -20.56,  -14.60,    2015
     c            13.33,   16.16,   11.76,    5.69,  -15.98,   -9.10,    2015
     d            -2.02,    2.26,    5.33,    8.83,  -21.77,    3.02,    2015
     e            10.76,   -3.22,   11.74,    0.67,   -6.74,  -13.20,    2015
     f            -6.88,   -0.10,    7.79,    8.68,    1.04,   -9.06,    2015
     g            -3.89,  -10.54,    8.44,   -2.01,   -6.26,    3.28,    2015
     h             0.17,   -0.40,    0.55,    4.55,   -0.55,    4.40,    2015
     i             1.70,   -7.92,   -0.67,   -0.61,    2.13,   -4.16,    2015
     j             2.33,   -2.85,   -1.80,   -1.12,   -3.59,   -8.72,    2015
     k             3.00,   -1.40,    0.00,   -2.30,    2.11,    2.08,    2015
     l            -0.60,   -0.79,   -1.05,    0.58,    0.76,   -0.70,    2015
     m            -0.20,    0.14,   -2.12,    1.70,   -1.44,   -0.22,    2015
     n            -2.57,    0.44,   -2.01,    3.49,   -2.34,   -2.09,    2015
     o            -0.16,   -1.08,    0.46,    0.37,    1.23,    1.75,    2015
     p            -0.89,   -2.19,    0.85,    0.27,    0.10,    0.72,    2015
     q             0.54,   -0.09,   -0.37,    0.29,   -0.43,    0.23,    2015
     r             0.22,   -0.89,   -0.94,   -0.16,   -0.03,    0.72,    2015
     s            -0.02,   -0.92,   -0.88,    0.42,    0.49,    0.63,    2015
     t             1.56,   -0.42,   -0.50,    0.96,   -1.24,   -0.19,    2015
     u            -0.10,    0.81,    0.42,   -0.13,   -0.04,    0.38,    2015
     v             0.48,    0.08,    0.48,    0.46,   -0.30,   -0.35,    2015
     w            -0.43,   -0.36,   -0.71/                               2015
      DATA GR/-29403.41,-1451.37, 4653.35,-2499.78, 2981.96,-2991.72,    2020
     1          1676.85, -734.62, 1363.00,-2380.80,  -81.96, 1236.06,    2020
     2           241.80,  525.60, -542.52,  902.82,  809.47,  282.10,    2020
     3            86.18, -158.50, -309.47,  199.75,   47.44, -350.30,    2020
     4          -234.42,  363.26,   47.52,  187.86,  208.36, -140.73,    2020
     5          -121.43, -151.16,   32.09,   13.98,   99.14,   65.97,    2020
     6            65.56,  -19.22,   72.96,   25.02, -121.57,   52.76,    2020
     7           -36.06,  -64.40,   13.60,    8.96,  -64.80,   68.04,    2020
     8            80.54,  -76.63,  -51.50,   -8.23,  -16.85,   56.45,    2020
     9             2.36,   15.80,   23.56,    6.30,   -2.19,   -7.21,    2020
     a           -27.19,    9.77,   -1.90,   23.66,    9.74,    8.43,    2020
     b           -17.49,  -15.23,   -0.49,   12.83,  -21.07,  -11.76,    2020
     c            15.28,   14.94,   13.65,    3.62,  -16.59,   -6.90,    2020
     d            -0.34,    2.90,    5.03,    8.36,  -23.44,    2.84,    2020
     e            11.04,   -1.48,    9.86,   -1.14,   -5.13,  -13.22,    2020
     f            -6.20,    1.08,    7.79,    8.82,    0.40,   -9.23,    2020
     g            -1.44,  -11.86,    9.60,   -1.84,   -6.25,    3.38,    2020
     h            -0.11,   -0.18,    1.66,    3.50,   -0.86,    4.86,    2020
     i             0.65,   -8.62,   -0.88,   -0.11,    1.88,   -4.26,    2020
     j             1.44,   -3.43,   -2.38,   -0.10,   -3.84,   -8.84,    2020
     k             2.96,   -1.36,   -0.02,   -2.51,    2.50,    2.31,    2020
     l            -0.55,   -0.85,   -0.39,    0.28,    0.62,   -0.66,    2020
     m            -0.21,   -0.07,   -1.66,    1.44,   -1.60,   -0.59,    2020
     n            -2.98,    0.18,   -1.97,    3.09,   -2.51,   -2.00,    2020
     o            -0.13,   -1.15,    0.43,    0.52,    1.28,    1.37,    2020
     p            -1.14,   -1.81,    0.71,    0.08,    0.31,    0.71,    2020
     q             0.49,   -0.15,   -0.26,    0.55,   -0.47,    0.16,    2020
     r             0.09,   -0.93,   -1.13,   -0.04,   -0.33,    0.52,    2020
     s             0.08,   -0.93,   -0.88,    0.53,    0.64,    0.72,    2020
     t             1.40,   -0.30,   -0.38,    0.75,   -1.31,   -0.01,    2020
     u            -0.09,    0.76,    0.29,   -0.05,   -0.11,    0.37,    2020
     v             0.47,    0.13,    0.54,    0.45,   -0.41,   -0.46,    2020
     w            -0.36,   -0.40,   -0.60/                               2020
      DATA GS/ -29350.0, -1410.3,  4545.5, -2556.2,  2950.9, -3133.6,    2025
     1           1648.7,  -814.2,  1360.9, -2404.2,   -56.9,  1243.8,    2025
     2            237.6,   453.4,  -549.6,   894.7,   799.6,   278.6,    2025
     3             55.8,  -134.0,  -281.1,   212.0,    12.0,  -375.4,    2025
     4           -232.9,   369.0,    45.3,   187.2,   220.0,  -138.7,    2025
     5           -122.9,  -141.9,    42.9,    20.9,   106.2,    64.3,    2025
     6             63.8,   -18.4,    76.7,    16.8,  -115.7,    48.9,    2025
     7            -40.9,   -59.8,    14.9,    10.9,   -60.8,    72.8,    2025
     8             79.6,   -76.9,   -48.9,    -8.8,   -14.4,    59.3,    2025
     9             -1.0,    15.8,    23.5,     2.5,    -7.4,   -11.2,    2025
     a            -25.1,    14.3,    -2.2,    23.1,    10.9,     7.2,    2025
     b            -17.5,   -12.6,     2.0,    11.5,   -21.8,    -9.7,    2025
     c             16.9,    12.7,    14.9,     0.7,   -16.8,    -5.2,    2025
     d              1.0,     3.9,     4.7,     8.0,   -24.8,     3.0,    2025
     e             12.1,    -0.2,     8.3,    -2.5,    -3.4,   -13.1,    2025
     f             -5.3,     2.4,     7.2,     8.6,    -0.6,    -8.7,    2025
     g              0.8,   -12.8,     9.8,    -1.3,    -6.4,     3.3,    2025
     h              0.2,     0.1,     2.0,     2.5,    -1.0,     5.4,    2025
     i             -0.5,    -9.0,    -0.9,     0.4,     1.5,    -4.2,    2025
     j              0.9,    -3.8,    -2.6,     0.9,    -3.9,    -9.0,    2025
     k              3.0,    -1.4,     0.0,    -2.5,     2.8,     2.4,    2025
     l             -0.6,    -0.6,     0.1,     0.0,     0.5,    -0.6,    2025
     m             -0.3,    -0.1,    -1.2,     1.1,    -1.7,    -1.0,    2025
     n             -2.9,    -0.1,    -1.8,     2.6,    -2.3,    -2.0,    2025
     o             -0.1,    -1.2,     0.4,     0.6,     1.2,     1.0,    2025
     p             -1.2,    -1.5,     0.6,     0.0,     0.5,     0.6,    2025
     q              0.5,    -0.2,    -0.1,     0.8,    -0.5,     0.1,    2025
     r             -0.2,    -0.9,    -1.2,     0.1,    -0.7,     0.2,    2025
     s              0.2,    -0.9,    -0.9,     0.6,     0.7,     0.7,    2025
     t              1.2,    -0.2,    -0.3,     0.5,    -1.3,     0.1,    2025
     u             -0.1,     0.7,     0.2,     0.0,    -0.2,     0.3,    2025
     v              0.5,     0.2,     0.6,     0.4,    -0.6,    -0.5,    2025
     w             -0.3,    -0.4,    -0.5/                               2025
      DATA GT/     12.6,    10.0,   -21.5,   -11.2,    -5.3,   -27.3,    2027
     1             -8.3,   -11.1,    -1.5,    -4.4,     3.8,     0.4,    2027
     2             -0.2,   -15.6,    -3.9,    -1.7,    -2.3,    -1.3,    2027
     3             -5.8,     4.1,     5.4,     1.6,    -6.8,    -4.1,    2027
     4              0.6,     1.3,    -0.5,     0.0,     2.1,     0.7,    2027
     5              0.5,     2.3,     1.7,     1.0,     1.9,    -0.2,    2027
     6             -0.3,     0.3,     0.8,    -1.6,     1.2,    -0.4,    2027
     7             -0.8,     0.8,     0.4,     0.7,     0.9,     0.9,    2027
     8             -0.1,    -0.1,     0.6,    -0.1,     0.5,     0.5,    2027
     9             -0.7,    -0.1,     0.0,    -0.8,    -0.9,    -0.8,    2027
     a              0.5,     0.9,    -0.3,    -0.1,     0.2,    -0.3,    2027
     b              0.0,     0.4,     0.4,    -0.3,    -0.1,     0.4,    2027
     c              0.3,    -0.5,     0.1,    -0.6,     0.0,     0.3,    2027
     d              0.3,     0.2, 115*0.0/                               2027
C-----------------------------------------------------------------------
C
C     Set initial values
C
      X     = 0.0D0
      Y     = 0.0D0
      Z     = 0.0D0
c
C     error return if date out of bounds
c
      IF ((DATE.LT.1900.0) .OR. (DATE.GT.2035.0)) THEN
         F     = 1.0D8
         WRITE (MSGTXT,1010) DATE
         CALL MSGWRT (8)
         GO TO 999
         END IF
C                                       marginal results > 2025
      IF ((DATE.GT.2030.0) .AND. (NMSG.LT.1)) THEN
         WRITE (MSGTXT,1015)
         CALL MSGWRT (6)
         WRITE (MSGTXT,1016) DATE
         CALL MSGWRT (6)
         WRITE (MSGTXT,1017)
         CALL MSGWRT (6)
         NMSG = NMSG + 1
         END IF
C                                       date >= 2025
      IF (DATE.GE.2025.0) THEN
         T     = DATE - 2025.0
         TC    = 1.0
         IF (ISV.EQ.1) THEN
            T = 1.0
            TC = 0.0
            END IF
C
C   pointer for last coefficient in pen-ultimate set of MF coefficients.
c
         LL    = 3450
         NMX   = 13
         NC    = NMX*(NMX+2)
         KMX   = (NMX+1)*(NMX+2)/2
C                                       date < 2025
      ELSE
         T     = 0.2*(DATE - 1900.0)
         LL    = T
         ONE   = LL
         T     = T - ONE
c
C     SH models before 1995.0 are only to degree 10
c
         IF (DATE.LT.1995.0) then
            NMX   = 10
            NC    = NMX*(NMX+2)
            LL    = NC*LL
            KMX   = (NMX+1)*(NMX+2)/2
         ELSE
            NMX   = 13
            NC    = NMX*(NMX+2)
            LL    = 0.2*(DATE - 1995.0)
C
C     19 is the number of SH models that extend to degree 10
c
            LL    = 120*19 + NC*LL
            KMX   = (NMX+1)*(NMX+2)/2
            END IF
         TC    = 1.0 - T
         IF (ISV.EQ.1) THEN
            TC = -0.2
            T = 0.2
            END IF
         END IF
C
      R     = ALT
      ONE   = COLAT*0.017453292
      CT    = COS(ONE)
      ST    = SIN(ONE)
      ONE   = ELONG*0.017453292
      CL(1) = COS(ONE)
      SL(1) = SIN(ONE)
      CD    = 1.0
      SD    = 0.0
      L     = 1
      M     = 1
      N     = 0
c
C     conversion from geodetic to geocentric coordinates
C     (using the WGS84 spheroid)
c
      IF (ITYPE.NE.2) THEN
         A2    = 40680631.6D0
         B2    = 40408296.0D0
         ONE   = A2 * ST * ST
         TWO   = B2 * CT * CT
         THREE = ONE + TWO
         RHO   = SQRT (THREE)
         R     = SQRT (ALT*(ALT + 2.0*RHO) + (A2*ONE + B2*TWO)/THREE)
         CD    = (ALT + RHO) / R
         SD    = (A2 - B2) / RHO * CT * ST / R
         ONE   = CT
         CT    = CT*CD -  ST*SD
         ST    = ST*CD + ONE*SD
         END IF
c
      RATIO = 6371.2D0 / R
      RR    = RATIO * RATIO
c
C     computation of Schmidt quasi-normal coefficients p and x(=q)
c
      P(1)  = 1.0
      P(3)  = ST
      Q(1)  = 0.0
      Q(3)  =  CT
      DO 10 K = 2,KMX
         IF (N.LT.M) THEN
            M     = 0
            N     = N + 1
            RR    = RR * RATIO
            FN    = N
            GN    = N - 1
            END IF
         FM    = M
         IF (M.NE.N) THEN
            GMM    = M * M
            ONE   = SQRT (FN*FN - GMM)
            TWO   = SQRT (GN*GN - GMM) / ONE
            THREE = (FN + GN) / ONE
            I     = K - N
            J     = I - N + 1
            P(K)  = THREE*CT*P(I) - TWO*P(J)
            Q(K)  = THREE*(CT*Q(I) - ST*P(I)) - TWO*Q(J)
         ELSE IF (K.NE.3) THEN
            ONE   = SQRT (1.0 - 0.5/FM)
            J     = K - N - 1
            P(K)  = ONE * ST * P(J)
            Q(K)  = ONE * (ST*Q(J) + CT*P(J))
            CL(M) = CL(M-1)*CL(1) - SL(M-1)*SL(1)
            SL(M) = SL(M-1)*CL(1) + CL(M-1)*SL(1)
            END IF
c
C     synthesis of x, y and z in geocentric coordinates
c
         LM    = LL + L
         ONE   = (TC*GH(LM) + T*GH(LM+NC)) * RR
         IF (M.EQ.0) THEN
            X     = X + ONE*Q(K)
            Z     = Z - (FN + 1.0)*ONE*P(K)
            L     = L + 1
         ELSE
            TWO   = (TC*GH(LM+1) + T*GH(LM+NC+1))*RR
            THREE = ONE*CL(M) + TWO*SL(M)
            X     = X + THREE*Q(K)
            Z     = Z - (FN + 1.0)*THREE*P(K)
            IF (ST.NE.0.0) THEN
               Y     = Y + (ONE*SL(M) - TWO*CL(M))*FM*P(K)/ST
            ELSE
               Y     = Y + (ONE*SL(M) - TWO*CL(M))*Q(K)*CT
               END IF
            L     = L + 2
            END IF
         M     = M + 1
 10      CONTINUE
c
C     conversion to coordinate system specified by itype
c
      ONE   = X
      X     = X*CD +   Z*SD
      Z     = Z*CD - ONE*SD
      F     = SQRT (X*X + Y*Y + Z*Z)
c
 999  RETURN
C-----------------------------------------------------------------------
 1010 FORMAT ('MGRF13 WILL NOT WORK WITH A DATE OF', F9.3,'.  DATE MUST'
     *   ' BE IN THE RANGE 1900 - 2035')
 1015 FORMAT ('THIS VERSION OF THE IGRF (14) IS INTENDED FOR USE UP',
     *   ' TO 2030.0.')
 1016 FORMAT ('VALUES FOR',F9.3,' WILL BE COMPUTED')
 1017 FORMAT ('BUT MAY BE OF REDUCED ACCURACY')
      END
