      SUBROUTINE TEINI (OPCODE, BUFFER, DISK, CNO, VER, CATBLK, LUN,
     *   ITERNO, TEKOLS, TENUMV, RDATE, TECTYP, IERR)
C-----------------------------------------------------------------------
C! Create/open/initialize TECOR details table
C# EXT-util Calibration
C-----------------------------------------------------------------------
C;  Copyright (C) 2023-2024
C;  Associated Universities, Inc. Washington DC, USA.
C;
C;  This program is free software; you can redistribute it and/or
C;  modify it under the terms of the GNU General Public License as
C;  published by the Free Software Foundation; either version 2 of
C;  the License, or (at your option) any later version.
C;
C;  This program is distributed in the hope that it will be useful,
C;  but WITHOUT ANY WARRANTY; without even the implied warranty of
C;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
C;  GNU General Public License for more details.
C;
C;  You should have received a copy of the GNU General Public
C;  License along with this program; if not, write to the Free
C;  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge,
C;  MA 02139, USA.
C;
C;  Correspondence concerning AIPS should be addressed as follows:
C;         Internet email: aipsmail@nrao.edu.
C;         Postal address: AIPS Project Office
C;                         National Radio Astronomy Observatory
C;                         520 Edgemont Road
C;                         Charlottesville, VA 22903-2475 USA
C-----------------------------------------------------------------------
C   Creates and initializes TECOR details table
C   Inputs:
C      OPCODE   C*4      Operation code:
C                        'WRIT' = create/init for write or read
C                        'READ' = open for read only
C      DISK     I        Disk to use.
C      CNO      I        Catalog slot number
C      LUN      I        Logical unit number to use
C   Input/output
C      VER      I        TE file version
C      CATBLK   I(256)   Catalog header block.
C      RDATE    C*8      Observation date
C      TECTYP   C*8      TEC data file name
C   Output:
C      BUFFER   I(512)   I/O buffer and related storage, also defines
C                        file if open.
C      ITERNO   I        Next scan number, start of the file if 'READ',
C                        the last+1 if WRITE
C      TEKOLS   I(MAXTEC)   The column pointer array in order:
C                        Time, source, antenna, ...
C      TENUMV   I(MAXTEC)   Element count in each column.
C      IERR     I        Return error code, 0=>OK, else TABINI or TABIO
C                        error.
C-----------------------------------------------------------------------
      INTEGER   MAXTEC
      PARAMETER (MAXTEC = 16)
C
      INCLUDE 'INCS:PUVD.INC'
      CHARACTER OPCODE*4, RDATE*8, TECTYP*8
      INTEGER   BUFFER(*), DISK, CNO, VER, CATBLK(256), LUN, ITERNO,
     *   TEKOLS(MAXTEC), TENUMV(MAXTEC), IERR
C
      HOLLERITH HOLTMP(6)
      CHARACTER TTITLE*56, TITLE(MAXTEC)*24, UNITS(MAXTEC)*8, KEYW(2)*8
      INTEGER   NKEY, NREC, DATP(128,2), NCOL, NTT, DTYP(MAXTEC), NDATA,
     *   KLOCS(2), KEYVAL(4), KEYTYP(2), IPOINT, MSGSAV, I, NC, JERR,
     *   ITRIM, ITEMP(6)
      REAL      KEYVAR(12)
      LOGICAL   DOREAD, NEWFIL
      INCLUDE 'INCS:DMSG.INC'
      INCLUDE 'INCS:DDCH.INC'
      EQUIVALENCE (KEYVAL, KEYVAR), (HOLTMP, ITEMP)
      DATA NTT /56/
      DATA TTITLE /'AIPS TECOR INTERNAL DETAILS TABLE '/
      DATA DTYP /11, 14, 14, 7*12, 32, 4*12, 22/
      DATA TITLE /'TIME', 'SOURCE', 'ANTENNA', 'HOUR ANGLE', 'AZIMUTH',
     *   'ZENITH ANGLE', 'AZION', 'ZAION', 'DLON', 'DLAT', 'BFIELD',
     *   'TEPATH', 'PROJECTED B', 'TOTAL TEC', 'IFR', 'DISP DELAY'/
      DATA KEYW /'RDATE', 'TECRTYPE'/
      DATA UNITS /'DAYS', ' ', ' ', 7*'DEGREES', 'GAUSS', '^16 /M^2',
     *   'GAUSS', '^16 /M^2', 'RAD/M^2', 'SEC/M^2'/
C-----------------------------------------------------------------------
C                                       Check OPCODE
      DOREAD = OPCODE.EQ.'READ'
C                                       Open file
      NREC = 1
      NKEY = 2
      NDATA = MAXTEC
      CALL FILL (NDATA, 0, TEKOLS)
      CALL FILL (NDATA, 0, TENUMV)
C                                       Fill in types
      IF (.NOT.DOREAD) THEN
         NCOL = NDATA
         CALL COPY (NDATA, DTYP, DATP(1,2))
      ELSE
         NCOL = 0
         END IF
C                                       Create/open file
      CALL TABINI (OPCODE, 'TE', DISK, CNO, VER, CATBLK, LUN, NKEY,
     *   NREC, NCOL, DATP, BUFFER, IERR)
      IF (IERR.GT.0) THEN
         CALL TABERR (OPCODE, 'TABINI', 'TEINI', IERR)
         GO TO 990
         END IF
      NEWFIL = IERR.LT.0
      MSGSAV = MSGSUP
C                                       Get number of scans
      ITERNO = BUFFER(5) + 1
      IF (DOREAD) ITERNO = 1
      NKEY = 2
C                                       File created, initialize
      IF (NEWFIL) THEN
C                                       Col. labels.
         DO 10 I = 1,NCOL
            CALL CHR2H (24, TITLE(I), 1, ITEMP)
            CALL TABIO ('WRIT', 3, I, ITEMP, BUFFER, IERR)
            IF (IERR.GT.0) THEN
               CALL TABERR ('WRIT', 'TABIO', 'TEINI', IERR)
               GO TO 990
               END IF
C                                       Units
            CALL CHR2H (8, UNITS(I), 1, ITEMP)
            CALL TABIO ('WRIT', 4, I, ITEMP, BUFFER, IERR)
            IF (IERR.GT.0) THEN
               CALL TABERR ('WRIT', 'TABIO', 'TEINI', IERR)
               GO TO 990
               END IF
 10         CONTINUE
C                                       Fill in Table title
         CALL CHR2H (NTT, TTITLE, 1, BUFFER(101))
C                                       Set keyword values
C                                       obs date
         KLOCS(1) = 1
         KEYTYP(1) = 3
         CALL CHR2H (8, RDATE, 1, KEYVAL(1))
         KLOCS(2) = 3
         KEYTYP(2) = 3
         CALL CHR2H (8, TECTYP, 1, KEYVAL(3))
C                                       Only write if just created.
         CALL TABKEY ('WRIT', KEYW, NKEY, BUFFER, KLOCS, KEYVAL, KEYTYP,
     *      IERR)
         IF ((IERR.GE.1) .AND. (IERR.LE.20)) THEN
            CALL TABERR ('WRIT', 'TABKEY', 'TEINI', IERR)
            GO TO 990
            END IF
C                                       Read keywords
      ELSE
         MSGSUP = 32000
         CALL TABKEY ('READ', KEYW, NKEY, BUFFER, KLOCS, KEYVAL, KEYTYP,
     *      IERR)
         MSGSUP = MSGSAV
         IF ((IERR.GE.1) .AND. (IERR.LE.20)) THEN
            CALL TABERR ('READ', 'TABKEY', 'TEINI', IERR)
            GO TO 990
            END IF
C                                       Retrieve keyword values
C                                       RDATE
         IPOINT = KLOCS(1)
         IF (IPOINT.GT.0) CALL H2CHR (8, 1, KEYVAR(1), RDATE)
         IPOINT = KLOCS(2)
         IF (IPOINT.GT.0) CALL H2CHR (8, 1, KEYVAR(3), TECTYP)
         END IF
      IERR = 0
C                                      Get array indices
C                                      Cover your ass from FNDCOL -
C                                      close to flush the buffers and
C                                      then reopen.
      CALL TABIO ('CLOS', 0, IPOINT, KEYVAL, BUFFER, IERR)
      IF (IERR.GT.0) THEN
         CALL TABERR ('CLOS', 'TABIO', 'TEINI', IERR)
         GO TO 990
         END IF
      NKEY = 0
      CALL TABINI (OPCODE, 'TE', DISK, CNO, VER, CATBLK, LUN, NKEY,
     *   NREC, NCOL, DATP, BUFFER, IERR)
      IF (IERR.GT.0) THEN
         CALL TABERR (OPCODE, 'TABINI', 'TEINI', IERR)
         GO TO 990
         END IF
      CALL FNDCOL (NDATA, TITLE, 24, .TRUE., BUFFER, TEKOLS, JERR)
C                                      Get array indices and no. values
      DO 150 I = 1,NDATA
         IPOINT = TEKOLS(I)
         IF (IPOINT.GT.0) THEN
            TEKOLS(I) = DATP(IPOINT,1)
            TENUMV(I) = DATP(IPOINT,2) / 10
            IF (TENUMV(I).LE.0) THEN
               NC = ITRIM (TITLE(I))
               WRITE (MSGTXT,1100) TITLE(I)(:NC)
               CALL MSGWRT (6)
               END IF
         ELSE
            TEKOLS(I) = -1
            TENUMV(I) = 0
            NC = ITRIM (TITLE(I))
            WRITE (MSGTXT,1101) TITLE(I)(:NC)
            CALL MSGWRT (6)
            END IF
 150     CONTINUE
      GO TO 999
C                                       Error
 990  WRITE (MSGTXT,1990) OPCODE
      CALL MSGWRT (7)
C
 999  RETURN
C-----------------------------------------------------------------------
 1100 FORMAT ('TEINI: ''',A,''' COLUMN HAS NO VALUES')
 1101 FORMAT ('TEINI: ''',A,''' COLUMN NOT FOUND')
 1990 FORMAT ('TEINI: ERROR INITIALIZING TE TABLE FOR ',A)
      END
